/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.CubicCurve2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.CubicCurve2D.Float, maps the x1, y1, ctrlx1, ctrly1, 
 * ctrlx2, ctrly2, x2 and y2 values to float-precision datastore fields.
 */
public class CubicCurve2dFloatMapping extends SingleFieldMultiMapping
{

    private static final CubicCurve2D.Float sampleValue = new CubicCurve2D.Float();

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.FLOAT); // X1
        addDatastoreField(ClassNameConstants.FLOAT); // Y1
        addDatastoreField(ClassNameConstants.FLOAT); // CtrlX1
        addDatastoreField(ClassNameConstants.FLOAT); // CtrlY1
        addDatastoreField(ClassNameConstants.FLOAT); // CtrlX2
        addDatastoreField(ClassNameConstants.FLOAT); // CtrlY2
        addDatastoreField(ClassNameConstants.FLOAT); // X2
        addDatastoreField(ClassNameConstants.FLOAT); // Y2
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return CubicCurve2D.Float.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {    	
    	CubicCurve2D.Float cubicCurve = (CubicCurve2D.Float)value;
        if (cubicCurve == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setFloat(preparedStatement,exprIndex[0],cubicCurve.x1);
            getDataStoreMapping(1).setFloat(preparedStatement,exprIndex[1],cubicCurve.y1);
            getDataStoreMapping(2).setFloat(preparedStatement,exprIndex[2],cubicCurve.ctrlx1);
            getDataStoreMapping(3).setFloat(preparedStatement,exprIndex[3],cubicCurve.ctrly1);
            getDataStoreMapping(4).setFloat(preparedStatement,exprIndex[4],cubicCurve.ctrlx2);
            getDataStoreMapping(5).setFloat(preparedStatement,exprIndex[5],cubicCurve.ctrly2);
            getDataStoreMapping(6).setFloat(preparedStatement,exprIndex[6],cubicCurve.x2);
            getDataStoreMapping(7).setFloat(preparedStatement,exprIndex[7],cubicCurve.y2);
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }
        
        float x1 = getDataStoreMapping(0).getFloat(resultSet,exprIndex[0]); 
        float y1 = getDataStoreMapping(1).getFloat(resultSet,exprIndex[1]); 
        float ctrlx1 = getDataStoreMapping(2).getFloat(resultSet,exprIndex[2]); 
        float ctrly1 = getDataStoreMapping(3).getFloat(resultSet,exprIndex[3]); 
        float ctrlx2 = getDataStoreMapping(4).getFloat(resultSet,exprIndex[4]);
        float ctrly2 = getDataStoreMapping(5).getFloat(resultSet,exprIndex[5]); 
        float x2 = getDataStoreMapping(6).getFloat(resultSet,exprIndex[6]);
        float y2 = getDataStoreMapping(7).getFloat(resultSet,exprIndex[7]);
        return new CubicCurve2D.Float(x1, y1, ctrlx1, ctrly1, ctrlx2, ctrly2, x2, y2);
    }
}