/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 
 

Contributors:
2003 Andy Jefferson - coding standards and javadocs
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.sql.Timestamp;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.store.mapped.expression.LogicSetExpression;
import org.datanucleus.store.mapped.expression.QueryExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.SqlTemporalExpression;
import org.datanucleus.store.mapped.expression.SqlTimestampLiteral;

/**
 * SCO Mapping for an SQLTimestamp type.
 */
public class SqlTimestampMapping extends SingleFieldMapping
{
    private static java.sql.Timestamp mappingSampleValue = new java.sql.Timestamp(new java.util.Date().getTime());

    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return mappingSampleValue;
    }

    public Class getJavaType()
    {
        return java.sql.Timestamp.class;
    }

    /**
     * Method to return the default length of this type in the datastore.
     * java.sql.Timestamp requires 29 characters ("YYYY-MM-DD HH:MM:SS.FFFFFFFFF")
     * @param index The index position
     * @return The default length
     */
    public int getDefaultLength(int index)
    {
        if (datastoreMappings != null && datastoreMappings.length > 0 && datastoreMappings[0].isStringBased())
        {
            // String-based storage when persisted as "YYYY-MM-DD HH:MM:SS.FFFFFFFFF"
            return 29;
        }
        else
        {
            return super.getDefaultLength(index);
        }
    }

    /**
     * Accessor for the name of the java-type actually used when mapping the particular datastore
     * field. This java-type must have an entry in the datastore mappings.
     * @param index requested datastore field index.
     * @return the name of java-type for the requested datastore field.
     */
    public String getJavaTypeForDatastoreMapping(int index)
    {
        if (datastoreMappings != null && datastoreMappings.length > 0 && datastoreMappings[0].isStringBased())
        {
            // Use String as our java type
            return ClassNameConstants.JAVA_LANG_STRING;
        }
        else
        {
            return super.getJavaTypeForDatastoreMapping(index);
        }
    }

    public ScalarExpression newLiteral(QueryExpression qs, Object value)
    {
        return new SqlTimestampLiteral(qs, this, (Timestamp)value);
    }

    public ScalarExpression newScalarExpression(QueryExpression qs, LogicSetExpression te)
    {
        return new SqlTemporalExpression(qs, this, te);
    }
}