/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.Arc2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.Arc2D.Float, maps the x, y, width, height, start and extent values to float-precision
 * datastore fields.
 */
public class Arc2dFloatMapping extends SingleFieldMultiMapping
{

    private static final Arc2D.Float sampleValue = new Arc2D.Float();

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.INT);   // Type
        addDatastoreField(ClassNameConstants.FLOAT); // X
        addDatastoreField(ClassNameConstants.FLOAT); // Y
        addDatastoreField(ClassNameConstants.FLOAT); // Width
        addDatastoreField(ClassNameConstants.FLOAT); // Height
        addDatastoreField(ClassNameConstants.FLOAT); // Start
        addDatastoreField(ClassNameConstants.FLOAT); // Extent
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return Arc2D.Float.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
    	Arc2D.Float arc = (Arc2D.Float)value;
        if (arc == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setInt(preparedStatement,exprIndex[0],arc.getArcType());
            getDataStoreMapping(1).setFloat(preparedStatement,exprIndex[1],arc.x);
            getDataStoreMapping(2).setFloat(preparedStatement,exprIndex[2],arc.y);
            getDataStoreMapping(3).setFloat(preparedStatement,exprIndex[3],arc.width);
            getDataStoreMapping(4).setFloat(preparedStatement,exprIndex[4],arc.height);
            getDataStoreMapping(5).setFloat(preparedStatement,exprIndex[5],arc.start);
            getDataStoreMapping(6).setFloat(preparedStatement,exprIndex[6],arc.extent);
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }
        
        int type = getDataStoreMapping(0).getInt(resultSet,exprIndex[0]); 
        float x = getDataStoreMapping(1).getFloat(resultSet,exprIndex[1]); 
        float y = getDataStoreMapping(2).getFloat(resultSet,exprIndex[2]); 
        float width  = getDataStoreMapping(3).getFloat(resultSet,exprIndex[3]); 
        float height = getDataStoreMapping(4).getFloat(resultSet,exprIndex[4]);
        float start  = getDataStoreMapping(5).getFloat(resultSet,exprIndex[5]); 
        float extent = getDataStoreMapping(6).getFloat(resultSet,exprIndex[6]);
        return new Arc2D.Float(x, y, width, height, start, extent, type);
    }
}