/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
package org.apache.causeway.core.runtimeservices.factory;

import java.util.Optional;

import jakarta.annotation.Priority;
import jakarta.inject.Named;
import jakarta.inject.Provider;

import org.springframework.beans.factory.annotation.Qualifier;
import org.jspecify.annotations.Nullable;
import org.springframework.stereotype.Service;

import org.apache.causeway.applib.annotation.PriorityPrecedence;
import org.apache.causeway.applib.graph.tree.TreeNode;
import org.apache.causeway.applib.services.bookmark.Bookmark;
import org.apache.causeway.applib.services.factory.FactoryService;
import org.apache.causeway.applib.services.iactnlayer.InteractionService;
import org.apache.causeway.commons.internal.base._Casts;
import org.apache.causeway.commons.internal.exceptions._Exceptions;
import org.apache.causeway.core.config.environment.CausewaySystemEnvironment;
import org.apache.causeway.core.metamodel.facets.object.mixin.MixinFacet;
import org.apache.causeway.core.metamodel.facets.object.navchild.ObjectTreeAdapter;
import org.apache.causeway.core.metamodel.object.ManagedObject;
import org.apache.causeway.core.metamodel.services.objectlifecycle.ObjectLifecyclePublisher;
import org.apache.causeway.core.metamodel.spec.ObjectSpecification;
import org.apache.causeway.core.metamodel.specloader.SpecificationLoader;
import org.apache.causeway.core.runtimeservices.CausewayModuleCoreRuntimeServices;

import org.jspecify.annotations.NonNull;

/**
 * Default implementation of {@link FactoryService}.
 *
 * @since 2.0 {@index}
 */
@Service
@Named(CausewayModuleCoreRuntimeServices.NAMESPACE + ".FactoryServiceDefault")
@Priority(PriorityPrecedence.MIDPOINT)
@Qualifier("Default")
public record FactoryServiceDefault(
        InteractionService interactionService, // dependsOn
        Provider<SpecificationLoader> specificationLoaderProvider,
        CausewaySystemEnvironment causewaySystemEnvironment,
        Provider<ObjectLifecyclePublisher> objectLifecyclePublisherProvider)
implements FactoryService {

    private ObjectLifecyclePublisher objectLifecyclePublisher() { return objectLifecyclePublisherProvider.get(); }

    @Override
    public <T> T getOrCreate(final @NonNull Class<T> requiredType) {
        var spec = loadSpecElseFail(requiredType);
        if(spec.isInjectable()) {
            return get(requiredType);
        }
        return create(requiredType);
    }

    @Override
    public <T> T get(final @NonNull Class<T> requiredType) {
        return causewaySystemEnvironment.springContextHolder()
                .get(requiredType)
                .orElseThrow(()->_Exceptions.noSuchElement("not an injectable type %s", requiredType));
    }

    @Override
    public <T> T detachedEntity(final @NonNull Class<T> domainClass) {
        var entitySpec = loadSpecElseFail(domainClass);
        if(!entitySpec.isEntity()) {
            throw _Exceptions.illegalArgument("Class '%s' is not an entity", domainClass.getName());
        }
        return createObject(domainClass, entitySpec);
    }

    @Override
    public <T> T detachedEntity(final @NonNull T entityPojo) {
        var entityClass = entityPojo.getClass();
        var spec = loadSpecElseFail(entityClass);
        if(!spec.isEntity()) {
            throw _Exceptions.illegalArgument("Type '%s' is not recognized as an entity type by the framework.",
                    entityClass);
        }
        objectLifecyclePublisher().onPostCreate(ManagedObject.entity(spec, entityPojo, Optional.empty()));
        return entityPojo;
    }

    @Override
    public <T> T mixin(final @NonNull Class<T> mixinClass, final @NonNull Object mixee) {
        var mixinSpec = loadSpecElseFail(mixinClass);
        var mixinFacet = mixinSpec.getFacet(MixinFacet.class);
        if(mixinFacet == null) {
            throw _Exceptions.illegalArgument("Class '%s' is not a mixin",
                    mixinClass.getName());
        }
        if(mixinSpec.isAbstract()) {
            throw _Exceptions.illegalArgument("Cannot instantiate abstract type '%s' as a mixin",
                    mixinClass.getName());
        }
        var mixin = mixinFacet.instantiate(mixee);
        return _Casts.uncheckedCast(mixin);
    }

    @Override
    public <T> T viewModel(final @NonNull T viewModelPojo) {
        var viewModelClass = viewModelPojo.getClass();
        var spec = loadSpecElseFail(viewModelClass);
        if(!spec.isViewModel()) {
            throw _Exceptions.illegalArgument("Type '%s' is not recognized as a ViewModel by the framework.",
                    viewModelClass);
        }
        spec.viewmodelFacetElseFail().initialize(viewModelPojo);
        objectLifecyclePublisher().onPostCreate(ManagedObject.viewmodel(spec, viewModelPojo, Optional.empty()));
        return viewModelPojo;
    }

    @Override
    public <T> T viewModel(final @NonNull Class<T> viewModelClass, final @Nullable Bookmark bookmark) {
        var spec = loadSpecElseFail(viewModelClass);
        return createViewModelElseFail(viewModelClass, spec, Optional.ofNullable(bookmark));
    }

    @Override
    public <T> T create(final @NonNull Class<T> domainClass) {
        var spec = loadSpecElseFail(domainClass);
        if(spec.isInjectable()) {
            throw _Exceptions.illegalArgument(
                    "Class '%s' is managed by Spring, use get() instead", domainClass.getName());
        }
        if(spec.isViewModel()) {
            return createViewModelElseFail(domainClass, spec, Optional.empty());
        }
        if(spec.isEntity()) {
            return detachedEntity(domainClass);
        }
        // fallback to generic object creation
        return createObject(domainClass, spec);
    }

    // -- HELPER

    private ObjectSpecification loadSpecElseFail(final @NonNull Class<?> type) {
        return specificationLoaderProvider().get().specForTypeElseFail(type);
    }

    /** handles injection, post-construct and publishing */
    private <T> T createViewModelElseFail(
            final @NonNull Class<T> viewModelClass,
            final @NonNull ObjectSpecification objectSpecification,
            final @NonNull Optional<Bookmark> bookmarkIfAny) {
        return Optional.of(objectSpecification)
        .filter(ObjectSpecification::isViewModel)
        .<T>map(spec->{
            var viewModel = spec.viewmodelFacetElseFail().instantiate(spec, bookmarkIfAny);
            objectLifecyclePublisher().onPostCreate(viewModel);
            return _Casts.uncheckedCast(viewModel.getPojo());
        })
        .orElseThrow(()->_Exceptions.illegalArgument("Type '%s' is not recognized as a ViewModel by the framework.",
                viewModelClass));
    }

    /** handles injection and publishing, but probably not post-construct */
    private <T> T createObject(
            final @NonNull Class<?> type,
            final @NonNull ObjectSpecification spec) {
        var domainObject = spec.createObject();
        return _Casts.uncheckedCast(domainObject.getPojo());
    }

    @Override
    public <T> TreeNode<T> treeNode(T root) {
        return TreeNode.root(root, _Casts.uncheckedCast(new ObjectTreeAdapter(specificationLoaderProvider().get())));
    }

}
