/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.kinesis;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.cloudwatch.model.Datapoint;
import com.amazonaws.services.cloudwatch.model.Dimension;
import com.amazonaws.services.cloudwatch.model.GetMetricStatisticsRequest;
import com.amazonaws.services.cloudwatch.model.GetMetricStatisticsResult;
import com.amazonaws.services.kinesis.AmazonKinesis;
import com.amazonaws.services.kinesis.clientlibrary.lib.worker.InitialPositionInStream;
import com.amazonaws.services.kinesis.clientlibrary.types.UserRecord;
import com.amazonaws.services.kinesis.model.DescribeStreamSummaryRequest;
import com.amazonaws.services.kinesis.model.ExpiredIteratorException;
import com.amazonaws.services.kinesis.model.GetRecordsRequest;
import com.amazonaws.services.kinesis.model.GetRecordsResult;
import com.amazonaws.services.kinesis.model.GetShardIteratorRequest;
import com.amazonaws.services.kinesis.model.LimitExceededException;
import com.amazonaws.services.kinesis.model.ListShardsRequest;
import com.amazonaws.services.kinesis.model.ListShardsResult;
import com.amazonaws.services.kinesis.model.ProvisionedThroughputExceededException;
import com.amazonaws.services.kinesis.model.Shard;
import com.amazonaws.services.kinesis.model.ShardFilter;
import com.amazonaws.services.kinesis.model.ShardFilterType;
import com.amazonaws.services.kinesis.model.ShardIteratorType;
import com.amazonaws.services.kinesis.model.StreamDescriptionSummary;
import java.io.IOException;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.function.Supplier;
import org.apache.beam.sdk.io.kinesis.AWSClientsProvider;
import org.apache.beam.sdk.io.kinesis.GetKinesisRecordsResult;
import org.apache.beam.sdk.io.kinesis.KinesisClientThrottledException;
import org.apache.beam.sdk.io.kinesis.StartingPoint;
import org.apache.beam.sdk.io.kinesis.TransientKinesisException;
import org.apache.beam.sdk.util.BackOff;
import org.apache.beam.sdk.util.BackOffUtils;
import org.apache.beam.sdk.util.FluentBackoff;
import org.apache.beam.sdk.util.Sleeper;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.base.Preconditions;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableList;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Minutes;
import org.joda.time.ReadableDuration;
import org.joda.time.ReadableInstant;

class SimplifiedKinesisClient {
    private static final String KINESIS_NAMESPACE = "AWS/Kinesis";
    private static final String INCOMING_RECORDS_METRIC = "IncomingBytes";
    private static final int PERIOD_GRANULARITY_IN_SECONDS = 60;
    private static final String SUM_STATISTIC = "Sum";
    private static final String STREAM_NAME_DIMENSION = "StreamName";
    private static final int LIST_SHARDS_MAX_RESULTS = 1000;
    private static final Duration SPACING_FOR_TIMESTAMP_LIST_SHARDS_REQUEST_TO_NOT_EXCEED_TRIM_HORIZON = Duration.standardMinutes((long)5L);
    private static final int DESCRIBE_STREAM_SUMMARY_MAX_ATTEMPTS = 10;
    private static final Duration DESCRIBE_STREAM_SUMMARY_INITIAL_BACKOFF = Duration.standardSeconds((long)1L);
    private final AmazonKinesis kinesis;
    private final AmazonCloudWatch cloudWatch;
    private final Integer limit;
    private final Supplier<Instant> currentInstantSupplier;

    public SimplifiedKinesisClient(AmazonKinesis kinesis, AmazonCloudWatch cloudWatch, Integer limit) {
        this(kinesis, cloudWatch, limit, Instant::now);
    }

    SimplifiedKinesisClient(AmazonKinesis kinesis, AmazonCloudWatch cloudWatch, Integer limit, Supplier<Instant> currentInstantSupplier) {
        this.kinesis = (AmazonKinesis)Preconditions.checkNotNull((Object)kinesis, (Object)"kinesis");
        this.cloudWatch = (AmazonCloudWatch)Preconditions.checkNotNull((Object)cloudWatch, (Object)"cloudWatch");
        this.limit = limit;
        this.currentInstantSupplier = currentInstantSupplier;
    }

    public static SimplifiedKinesisClient from(AWSClientsProvider provider, Integer limit) {
        return new SimplifiedKinesisClient(provider.getKinesisClient(), provider.getCloudWatchClient(), limit);
    }

    public String getShardIterator(String streamName, String shardId, ShardIteratorType shardIteratorType, String startingSequenceNumber, Instant timestamp) throws TransientKinesisException {
        Date date = timestamp != null ? timestamp.toDate() : null;
        return this.wrapExceptions(() -> this.kinesis.getShardIterator(new GetShardIteratorRequest().withStreamName(streamName).withShardId(shardId).withShardIteratorType(shardIteratorType).withStartingSequenceNumber(startingSequenceNumber).withTimestamp(date)).getShardIterator());
    }

    public List<Shard> listShardsAtPoint(String streamName, StartingPoint startingPoint) throws TransientKinesisException {
        ShardFilter shardFilter = this.wrapExceptions(() -> this.buildShardFilterForStartingPoint(streamName, startingPoint));
        return this.listShards(streamName, shardFilter);
    }

    private ShardFilter buildShardFilterForStartingPoint(String streamName, StartingPoint startingPoint) throws IOException, InterruptedException {
        InitialPositionInStream position = startingPoint.getPosition();
        switch (position) {
            case LATEST: {
                return new ShardFilter().withType(ShardFilterType.AT_LATEST);
            }
            case TRIM_HORIZON: {
                return new ShardFilter().withType(ShardFilterType.AT_TRIM_HORIZON);
            }
            case AT_TIMESTAMP: {
                return this.buildShardFilterForTimestamp(streamName, startingPoint.getTimestamp());
            }
        }
        throw new IllegalArgumentException(String.format("Unrecognized '%s' position to create shard filter with", position));
    }

    private ShardFilter buildShardFilterForTimestamp(String streamName, Instant startingPointTimestamp) throws IOException, InterruptedException {
        StreamDescriptionSummary streamDescription = this.describeStreamSummary(streamName);
        Instant streamCreationTimestamp = new Instant((Object)streamDescription.getStreamCreationTimestamp());
        if (streamCreationTimestamp.isAfter((ReadableInstant)startingPointTimestamp)) {
            return new ShardFilter().withType(ShardFilterType.AT_TRIM_HORIZON);
        }
        Duration retentionPeriod = Duration.standardHours((long)streamDescription.getRetentionPeriodHours().intValue());
        Instant streamTrimHorizonTimestamp = this.currentInstantSupplier.get().minus((ReadableDuration)retentionPeriod).plus((ReadableDuration)SPACING_FOR_TIMESTAMP_LIST_SHARDS_REQUEST_TO_NOT_EXCEED_TRIM_HORIZON);
        if (startingPointTimestamp.isAfter((ReadableInstant)streamTrimHorizonTimestamp)) {
            return new ShardFilter().withType(ShardFilterType.AT_TIMESTAMP).withTimestamp(startingPointTimestamp.toDate());
        }
        return new ShardFilter().withType(ShardFilterType.AT_TRIM_HORIZON);
    }

    private StreamDescriptionSummary describeStreamSummary(String streamName) throws IOException, InterruptedException {
        FluentBackoff retryBackoff = FluentBackoff.DEFAULT.withMaxRetries(10).withInitialBackoff(DESCRIBE_STREAM_SUMMARY_INITIAL_BACKOFF);
        BackOff backoff = retryBackoff.backoff();
        Sleeper sleeper = Sleeper.DEFAULT;
        DescribeStreamSummaryRequest request = new DescribeStreamSummaryRequest();
        request.setStreamName(streamName);
        while (true) {
            try {
                return this.kinesis.describeStreamSummary(request).getStreamDescriptionSummary();
            }
            catch (LimitExceededException exc) {
                if (BackOffUtils.next((Sleeper)sleeper, (BackOff)backoff)) continue;
                throw exc;
            }
            break;
        }
    }

    public List<Shard> listShardsFollowingClosedShard(String streamName, String exclusiveStartShardId) throws TransientKinesisException {
        ShardFilter shardFilter = new ShardFilter().withType(ShardFilterType.AFTER_SHARD_ID).withShardId(exclusiveStartShardId);
        return this.listShards(streamName, shardFilter);
    }

    private List<Shard> listShards(String streamName, ShardFilter shardFilter) throws TransientKinesisException {
        return (List)this.wrapExceptions(() -> {
            ListShardsResult response;
            ImmutableList.Builder shardsBuilder = ImmutableList.builder();
            String currentNextToken = null;
            do {
                ListShardsRequest request = new ListShardsRequest();
                request.setMaxResults(Integer.valueOf(1000));
                if (currentNextToken != null) {
                    request.setNextToken(currentNextToken);
                } else {
                    request.setStreamName(streamName);
                }
                request.setShardFilter(shardFilter);
                response = this.kinesis.listShards(request);
                List shards = response.getShards();
                shardsBuilder.addAll((Iterable)shards);
            } while ((currentNextToken = response.getNextToken()) != null);
            return shardsBuilder.build();
        });
    }

    public GetKinesisRecordsResult getRecords(String shardIterator, String streamName, String shardId) throws TransientKinesisException {
        return this.getRecords(shardIterator, streamName, shardId, this.limit);
    }

    public GetKinesisRecordsResult getRecords(String shardIterator, String streamName, String shardId, Integer limit) throws TransientKinesisException {
        return this.wrapExceptions(() -> {
            GetRecordsResult response = this.kinesis.getRecords(new GetRecordsRequest().withShardIterator(shardIterator).withLimit(limit));
            return new GetKinesisRecordsResult(UserRecord.deaggregate((List)response.getRecords()), response.getNextShardIterator(), response.getMillisBehindLatest(), streamName, shardId);
        });
    }

    public long getBacklogBytes(String streamName, Instant countSince) throws TransientKinesisException {
        return this.getBacklogBytes(streamName, countSince, new Instant());
    }

    public long getBacklogBytes(String streamName, Instant countSince, Instant countTo) throws TransientKinesisException {
        return this.wrapExceptions(() -> {
            Minutes period = Minutes.minutesBetween((ReadableInstant)countSince, (ReadableInstant)countTo);
            if (period.isLessThan(Minutes.ONE)) {
                return 0L;
            }
            GetMetricStatisticsRequest request = this.createMetricStatisticsRequest(streamName, countSince, countTo, period);
            long totalSizeInBytes = 0L;
            GetMetricStatisticsResult result = this.cloudWatch.getMetricStatistics(request);
            for (Datapoint point : result.getDatapoints()) {
                totalSizeInBytes += point.getSum().longValue();
            }
            return totalSizeInBytes;
        });
    }

    GetMetricStatisticsRequest createMetricStatisticsRequest(String streamName, Instant countSince, Instant countTo, Minutes period) {
        return new GetMetricStatisticsRequest().withNamespace(KINESIS_NAMESPACE).withMetricName(INCOMING_RECORDS_METRIC).withPeriod(Integer.valueOf(period.getMinutes() * 60)).withStartTime(countSince.toDate()).withEndTime(countTo.toDate()).withStatistics(Collections.singletonList(SUM_STATISTIC)).withDimensions(Collections.singletonList(new Dimension().withName(STREAM_NAME_DIMENSION).withValue(streamName)));
    }

    private <T> T wrapExceptions(Callable<T> callable) throws TransientKinesisException {
        try {
            return callable.call();
        }
        catch (ExpiredIteratorException e) {
            throw e;
        }
        catch (LimitExceededException | ProvisionedThroughputExceededException e) {
            throw new KinesisClientThrottledException("Too many requests to Kinesis. Wait some time and retry.", (AmazonClientException)e);
        }
        catch (AmazonServiceException e) {
            if (e.getErrorType() == AmazonServiceException.ErrorType.Service) {
                throw new TransientKinesisException("Kinesis backend failed. Wait some time and retry.", (AmazonClientException)((Object)e));
            }
            throw new RuntimeException("Kinesis client side failure", e);
        }
        catch (AmazonClientException e) {
            if (e.isRetryable()) {
                throw new TransientKinesisException("Retryable client failure", e);
            }
            throw new RuntimeException("Not retryable client failure", e);
        }
        catch (Exception e) {
            throw new RuntimeException("Unknown kinesis failure, when trying to reach kinesis", e);
        }
    }
}

