/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ignite3.internal.eventlog.impl;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.NoSuchFileException;
import java.security.GeneralSecurityException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import org.apache.ignite3.internal.eventlog.api.Event;
import org.apache.ignite3.internal.eventlog.api.Sink;
import org.apache.ignite3.internal.eventlog.config.schema.WebhookSinkRetryPolicyView;
import org.apache.ignite3.internal.eventlog.config.schema.WebhookSinkView;
import org.apache.ignite3.internal.eventlog.ser.EventSerializer;
import org.apache.ignite3.internal.logger.IgniteLogger;
import org.apache.ignite3.internal.logger.Loggers;
import org.apache.ignite3.internal.network.configuration.SslView;
import org.apache.ignite3.internal.network.ssl.KeystoreLoader;
import org.apache.ignite3.internal.rest.constants.HttpCode;
import org.apache.ignite3.internal.thread.IgniteThreadFactory;
import org.apache.ignite3.internal.thread.ThreadOperation;
import org.apache.ignite3.internal.util.IgniteUtils;
import org.apache.ignite3.lang.ErrorGroups;
import org.apache.ignite3.lang.IgniteException;
import org.jetbrains.annotations.TestOnly;

class WebhookSink
implements Sink<WebhookSinkView> {
    private static final IgniteLogger LOG = Loggers.forClass(WebhookSink.class);
    private static final Set<Integer> RETRYABLE_STATUSES = Set.of(Integer.valueOf(HttpCode.TOO_MANY_REQUESTS.code()), Integer.valueOf(HttpCode.BAD_GATEWAY.code()), Integer.valueOf(HttpCode.SERVICE_UNAVAILABLE.code()), Integer.valueOf(HttpCode.GATEWAY_TIMEOUT.code()));
    private final WebhookSinkView cfg;
    private final EventSerializer serializer;
    private final Supplier<UUID> clusterIdSupplier;
    private final String nodeName;
    private final HttpClient client;
    private final BlockingQueue<Event> events;
    private final ScheduledExecutorService executorService;
    private long lastSendMillis;

    WebhookSink(WebhookSinkView cfg, EventSerializer serializer, Supplier<UUID> clusterIdSupplier, String nodeName) {
        this.cfg = cfg;
        this.serializer = serializer;
        this.clusterIdSupplier = clusterIdSupplier;
        this.nodeName = nodeName;
        this.events = new LinkedBlockingQueue<Event>(cfg.queueSize());
        this.client = WebhookSink.configureClient(cfg);
        this.executorService = Executors.newSingleThreadScheduledExecutor(IgniteThreadFactory.create(nodeName, "eventlog-webhook-sink", LOG, new ThreadOperation[0]));
        this.executorService.scheduleAtFixedRate(this::tryToSendBatch, cfg.batchSendFrequencyMillis(), cfg.batchSendFrequencyMillis(), TimeUnit.MILLISECONDS);
    }

    @Override
    public void stop() {
        if (this.executorService != null) {
            if (!this.events.isEmpty()) {
                this.executorService.execute(this::tryToSendBatch);
            }
            IgniteUtils.shutdownAndAwaitTermination(this.executorService, 1L, TimeUnit.SECONDS);
        }
        this.events.clear();
    }

    @Override
    public void write(Event event) {
        while (!this.events.offer(event)) {
            this.events.poll();
        }
        if (this.events.size() >= this.cfg.batchSize()) {
            this.executorService.execute(this::tryToSendBatch);
        }
    }

    @TestOnly
    BlockingQueue<Event> getEvents() {
        return this.events;
    }

    public long getLastSendMillis() {
        return this.lastSendMillis;
    }

    private void sendInternal(Collection<Event> batch) {
        WebhookSinkRetryPolicyView rp = this.cfg.retryPolicy();
        int retryCounter = 0;
        Throwable lastError = null;
        do {
            if (retryCounter > 0) {
                long currentBackoff = (long)((double)rp.initBackoffMillis() * Math.pow(rp.backoffMultiplier(), retryCounter));
                long backoff = Math.min(currentBackoff, rp.maxBackoffMillis());
                try {
                    Thread.sleep(backoff);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    break;
                }
            }
            try {
                HttpResponse<String> res = this.client.send(this.createRequest(this.serializer.serialize(batch)), HttpResponse.BodyHandlers.ofString(StandardCharsets.UTF_8));
                if (RETRYABLE_STATUSES.contains(res.statusCode())) {
                    LOG.trace("Failed to send events to webhook, will retry attempt [name={}, retry={}, statusCode={}]", this.cfg.endpoint(), retryCounter, res.statusCode());
                    continue;
                }
                LOG.trace("Successfully send events to webhook [name={}, eventCount={}]", this.cfg.endpoint(), batch.size());
                lastError = null;
                break;
            }
            catch (Throwable e) {
                LOG.trace("Failed to send events to webhook, will retry attempt [name={}, retry={}]", this.cfg.endpoint(), retryCounter, e);
                lastError = e;
            }
        } while (++retryCounter < rp.maxAttempts());
        if (lastError != null) {
            LOG.warn("Failed to send events to webhook [name={}, eventCount={} lastError={}]", this.cfg.endpoint(), batch.size(), lastError.getMessage());
        }
    }

    private void tryToSendBatch() {
        if (this.events.isEmpty()) {
            this.lastSendMillis = System.currentTimeMillis();
            return;
        }
        ArrayList<Event> batch = new ArrayList<Event>(this.cfg.batchSize());
        while (!(this.events.isEmpty() || this.events.size() < this.cfg.batchSize() && System.currentTimeMillis() - this.lastSendMillis <= this.cfg.batchSendFrequencyMillis())) {
            this.events.drainTo(batch, this.cfg.batchSize());
            this.sendInternal(batch);
            batch.clear();
            this.lastSendMillis = System.currentTimeMillis();
        }
    }

    private HttpRequest createRequest(byte[] body) {
        return HttpRequest.newBuilder(URI.create(this.cfg.endpoint())).header("Content-Type", "application/json").header("X-SINK-CLUSTER-ID", String.valueOf(this.clusterIdSupplier.get())).header("X-SINK-NODE-NAME", this.nodeName).POST(HttpRequest.BodyPublishers.ofByteArray(body)).build();
    }

    private static HttpClient configureClient(WebhookSinkView cfg) {
        HttpClient.Builder builder = HttpClient.newBuilder();
        if (cfg.ssl().enabled()) {
            builder.sslContext(WebhookSink.createClientSslContext(cfg.ssl(), WebhookSink.createTrustManagerFactory(cfg.ssl())));
        }
        return builder.build();
    }

    private static TrustManagerFactory createTrustManagerFactory(SslView ssl) {
        try {
            TrustManagerFactory trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init(KeystoreLoader.load(ssl.trustStore()));
            return trustManagerFactory;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IgniteException(ErrorGroups.Common.SSL_CONFIGURATION_ERR, (Throwable)e);
        }
    }

    private static SSLContext createClientSslContext(SslView ssl, TrustManagerFactory trustManagerFactory) {
        try {
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(KeystoreLoader.load(ssl.keyStore()), ssl.keyStore().password().toCharArray());
            SSLContext sslContext = SSLContext.getInstance("TLS");
            sslContext.init(keyManagerFactory.getKeyManagers(), trustManagerFactory.getTrustManagers(), new SecureRandom());
            return sslContext;
        }
        catch (NoSuchFileException e) {
            throw new IgniteException(ErrorGroups.Common.SSL_CONFIGURATION_ERR, String.format("File %s not found", e.getMessage()), (Throwable)e);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new IgniteException(ErrorGroups.Common.SSL_CONFIGURATION_ERR, (Throwable)e);
        }
    }
}

