/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.utils;

import static org.apache.james.utils.NamingScheme.IDENTITY;
import static org.assertj.core.api.Assertions.assertThat;

import org.junit.jupiter.api.Nested;
import org.junit.jupiter.api.Test;

class NamingSchemeTest {
    public static final String PACKAGE = "org.apache.james";
    public static final NamingScheme.OptionalPackagePrefix PACKAGE_PREFIX = new NamingScheme.OptionalPackagePrefix(PackageName.of(PACKAGE));

    @Nested
    class IdentityTest {
        @Test
        void asFullyQualifiedShouldPromoteNameToFullyQualifiedClass() {
            String name = "org.MyClass";
            ClassName className = new ClassName(name);

            assertThat(IDENTITY.toFullyQualifiedClassNames(className))
                .containsExactly(new FullyQualifiedClassName(name));
        }

        @Test
        void asFullyQualifiedShouldPromoteNameToFullyQualifiedClassWhenNoPackagePart() {
            String name = "MyClass";
            ClassName className = new ClassName(name);

            assertThat(IDENTITY.toFullyQualifiedClassNames(className))
                .containsExactly(new FullyQualifiedClassName(name));
        }
    }

    @Nested
    class OptionalPackagePrefixTest {

        @Test
        void asFullyQualifiedShouldPromoteNameToFullyQualifiedClass() {
            String name = "org.MyClass";
            ClassName className = new ClassName(name);

            assertThat(PACKAGE_PREFIX.toFullyQualifiedClassNames(className))
                .containsExactly(
                    new FullyQualifiedClassName("org.MyClass"),
                    new FullyQualifiedClassName("org.apache.james.org.MyClass"));
        }

        @Test
        void asFullyQualifiedShouldPromoteNameToFullyQualifiedClassWhenNoPackagePart() {
            String name = "MyClass";
            ClassName className = new ClassName(name);

            assertThat(PACKAGE_PREFIX.toFullyQualifiedClassNames(className))
                .containsExactly(
                    new FullyQualifiedClassName("MyClass"),
                    new FullyQualifiedClassName("org.apache.james.MyClass"));
        }
    }
}