/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.lib;

/**
 * <p>
 * Internal helper class for some class management.
 * </p>
 * <p>
 * <b>Not for use outside of Orchestra</b>
 * </p>
 */
public final class _ClassUtils
{
    private _ClassUtils()
    {
    }

    /**
     * Create a new instance for a class by its name
     * 
     * @since 1.1
     */
    public static Object newInstance(Class clazz)
    {
        try
        {
            return clazz.newInstance();
        }
        catch(NoClassDefFoundError e)
        {
            throw new OrchestraException(e);
        }
        catch (InstantiationException e)
        {
            throw new OrchestraException(e);
        }
        catch (IllegalAccessException e)
        {
            throw new OrchestraException(e);
        }
    }

    /**
     * create a new instance for a class by its name
     */
    public static Object newInstance(String className)
    {
        try
        {
            Class clazz = classForName(className);
            return clazz.newInstance();
        }
        catch(NoClassDefFoundError e)
        {
            throw new OrchestraException(e);
        }
        catch (InstantiationException e)
        {
            throw new OrchestraException(e);
        }
        catch (IllegalAccessException e)
        {
            throw new OrchestraException(e);
        }
        catch (ClassNotFoundException e)
        {
            throw new OrchestraException(e);
        }
    }

    /**
     * Lookup class using the webapp classloader first and the the classloader which loaded
     * this class.
     */
    public static Class classForName(String className) throws ClassNotFoundException
    {
        if (className == null)
        {
            throw new NullPointerException("className");
        }

        try
        {
            // Try WebApp ClassLoader first
            return Class.forName(className,
                false, // do not initialize for faster startup
                Thread.currentThread().getContextClassLoader());
        }
        catch (ClassNotFoundException ignore)
        {
            // fallback: Try ClassLoader for this class (i.e. the myfaces.jar lib)
            return Class.forName(className,
                false, // do not initialize for faster startup
                _ClassUtils.class.getClassLoader());
        }
    }
}
