/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation.spring;

import org.apache.myfaces.orchestra.conversation.Conversation;
import org.apache.myfaces.orchestra.conversation.ConversationManager;
import org.apache.myfaces.orchestra.conversation.SimpleBean;
import org.apache.myfaces.orchestra.conversation.basic.LogConversationMessager;
import org.apache.myfaces.orchestra.frameworkAdapter.FrameworkAdapter;
import org.apache.myfaces.orchestra.frameworkAdapter.local.LocalFrameworkAdapter;
import org.springframework.aop.SpringProxy;
import org.springframework.aop.scope.ScopedObject;
import org.springframework.test.AbstractDependencyInjectionSpringContextTests;


/**
 * Test various methods on the _SpringUtils class.
 */
public class TestSpringUtils extends AbstractDependencyInjectionSpringContextTests
{
    protected String[] getConfigLocations()
    {
        return new String[]
            {
                "classpath:org/apache/myfaces/orchestra/conversation/spring/TestSpringUtils.xml"
            };
    }

    protected void onSetUp() throws Exception
    {
        super.onSetUp();

        LocalFrameworkAdapter frameworkAdapter = new LocalFrameworkAdapter();
        frameworkAdapter.setApplicationContext(applicationContext);
        frameworkAdapter.setConversationMessager(new LogConversationMessager());
        FrameworkAdapter.setCurrentInstance(frameworkAdapter);
    }

    protected void onTearDown() throws Exception
    {
        FrameworkAdapter.setCurrentInstance(null);
        super.onTearDown();
    }

    public void testConversation() throws Exception
    {
        final String BEAN_NAME = "simpleBean";

        // The Spring configuration for dummyBean does not explicitly set a conversation name,
        // so conversation-name = bean-name
        final String CONVERSATION_NAME = BEAN_NAME;

        // create a scoped-proxy (but not the underlying bean)
        SimpleBean scopedProxy = (SimpleBean) applicationContext.getBean(BEAN_NAME);

        assertTrue("should be a scoped object", scopedProxy instanceof ScopedObject);
        assertTrue("should be a proxy", scopedProxy instanceof SpringProxy);
        assertFalse("conversation should not have been started yet", ConversationManager.getInstance().hasConversation(CONVERSATION_NAME));

        // Invoke a method on the scoped-proxy, forcing the conversation-proxy to be created, which will in turn
        // force the real bean to be created. The name "conversation-proxy" means the proxy to which the advices
        // are attached, including the CurrentConversationAdvice which sets up the conversation before the real
        // bean is invoked.
        scopedProxy.getData();
        assertTrue("conversation should have been started", ConversationManager.getInstance().hasConversation(CONVERSATION_NAME));

        // Obtain a reference to the conversation-proxy bean. The scoped-proxy and conversation-proxy will be different objects.
        SimpleBean convProxy = (SimpleBean) _SpringUtils.getTargetObject(scopedProxy);
        assertTrue("should be a proxy", convProxy instanceof SpringProxy);
        assertTrue(convProxy != scopedProxy);

        // Just for fun, obtain a reference to the real actual bean too.
        SimpleBean realBean = (SimpleBean) _SpringUtils.getTargetObject(convProxy);
        assertFalse("should not be a proxy", realBean instanceof SpringProxy);
        assertTrue(realBean != convProxy);

        // Check that the scoped-proxy maps all method-calls to the conversation-proxy which in turn maps the
        // real object by modifying data on the underlying object then reading  via the indirect reference, and
        // vice-versa
        assertEquals(null, convProxy.getData());
        scopedProxy.setData("proxy");
        assertEquals("proxy", realBean.getData());
        realBean.setData("real");
        assertEquals("real", scopedProxy.getData());

        // After invalidating the conversation and recreating the bean, the scopedProxy no longer refers
        // to the same object.
        Conversation conv = ConversationManager.getInstance().getConversation(CONVERSATION_NAME);
        conv.invalidate();
        assertEquals(null, scopedProxy.getData()); // this is a new object with reset data property
        SimpleBean newConvProxy = (SimpleBean) _SpringUtils.getTargetObject(scopedProxy);
        assertNotSame(convProxy, newConvProxy);

        // Writing via the proxy no longer touches the original realBean
        scopedProxy.setData("proxy");
        assertTrue("proxy".equals(scopedProxy.getData()));
        assertFalse("proxy".equals(realBean.getData()));
        
        // And accessing the old proxy is no longer allowed. Note that this condition isn't normally
        // possible in user code as users don't use non-public api _SpringUtils.getTargetObject to
        // unwrap proxies.
        try
        {
            convProxy.getData();
            fail("old proxy still accessable after conversation has terminated");
        }
        catch(IllegalStateException e)
        {
            // ok, expected
        }
    }
}
