/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.viewController;

import org.apache.myfaces.orchestra.annotation.AnnotationInfo;
import org.apache.myfaces.orchestra.annotation.AnnotationInfoManager;
import org.apache.myfaces.orchestra.lib.OrchestraException;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Execute the various viewController events on the viewController by calling
 * the corresponding annotated method.
 */
public class AnnotationsViewControllerExecutor extends AbstractViewControllerExecutor
{
    private final AnnotationInfoManager annotationInfoManager;

    public AnnotationsViewControllerExecutor(AnnotationInfoManager annotationInfoManager)
    {
        this.annotationInfoManager = annotationInfoManager;
    }

    protected void invokeMethod(Object bean, Method method)
    {
        try
        {
            method.invoke(bean, (Object[]) null);
        }
        catch (IllegalAccessException e)
        {
            throw new OrchestraException(e);
        }
        catch (InvocationTargetException e)
        {
            throw new OrchestraException(e);
        }
    }

    public boolean invokeInitView(String beanName, Object bean)
    {
        AnnotationInfo annotationsInfo = annotationInfoManager.getAnnotationInfoByBeanName(beanName);
        if (annotationsInfo != null)
        {
            Method method = annotationsInfo.getInitViewMethod();
            if (method != null)
            {
                invokeMethod(bean, method);
                return true;
            }
        }
        return false;
    }

    public boolean invokePreRenderView(String beanName, Object bean)
    {
        AnnotationInfo annotationsInfo = annotationInfoManager.getAnnotationInfoByBeanName(beanName);
        if (annotationsInfo != null)
        {
            Method method = annotationsInfo.getPreRenderViewMethod();
            if (method != null)
            {
                invokeMethod(bean, method);
                return true;
            }
        }
        return false;
    }

    public boolean invokePreProcess(String beanName, Object bean)
    {
        AnnotationInfo annotationsInfo = annotationInfoManager.getAnnotationInfoByBeanName(beanName);
        if (annotationsInfo != null)
        {
            Method method = annotationsInfo.getPreProcessMethod();
            if (method != null)
            {
                invokeMethod(bean, method);
                return true;
            }
        }
        return false;
    }
}
