##########################################################
####### Single Starter/Generator electrical system #######
####################    Syd Adams    #####################
###### Based on Curtis Olson's nasal electrical code #####
##########################################################
##########################################################
######## Modified by Clement DE L'HAMAIDE for P92 ########
##########################################################
######## Modified by Bea Wolf			  ########
######## for CH750 2020-2023				  ########
##########################################################

# References:
#		http://www.aircraft-battery.com/search-by-your-aircraft/battery_detail/156
#		http://www.zenithair.net/wp-content/uploads/stolch750/data/Draft%20POH%20STOL%20CH%20750-SLSA%20June%202009.pdf

var last_time = 0.0;
var OutPuts = props.globals.getNode("/systems/electrical/outputs",1);
var Volts = props.globals.getNode("/systems/electrical/volts",1);
var Amps = props.globals.getNode("/systems/electrical/amps",1);
var delta_t = props.globals.getNode("/sim/time/delta-sec");
var eng_amp = props.globals.initNode("/engines/engine/amp-v", 0.0, "DOUBLE");

var cb = props.globals.getNode("/controls/circuit-breakers");

var switches = {
	bat:			props.globals.getNode("/controls/engines/engine/master-bat",1),
	alt:			props.globals.getNode("/controls/engines/engine/master-alt",1),
	starter:		props.globals.getNode("/controls/engines/engine[0]/starter", 1),
	
	radio_master:	props.globals.initNode("/controls/switches/radio-master",	0,	"BOOL"),
	strobe:		props.globals.initNode("/controls/lighting/strobe",		0,	"BOOL"),
	nav:			props.globals.initNode("/controls/lighting/nav-lights",	0,	"BOOL"),
	landing:		props.globals.initNode("/controls/lighting/landing-light",	0,	"BOOL"),
	boost:		props.globals.initNode("/controls/engines/engine/fuel-pump",0,	"BOOL"),
};

var serviceable = {
	electric:	props.globals.getNode("systems/electrical/serviceable",1),
	comm:		props.globals.getNode("/instrumentation/comm/serviceable"),
	xpdr:		props.globals.getNode("instrumentation/transponder/serviceable"),
};

var circuit_breakers = {
	nav_lts:	cb.initNode("nav-lights",	1,	"BOOL"),
	strobe_lts:	cb.initNode("strobe-lights",	1,	"BOOL"),
	radio: [
		cb.initNode("radio1",	1,	"BOOL"),
		cb.initNode("radio2",	1,	"BOOL"),
	],
	gps:		cb.initNode("gps",		1,	"BOOL"),
	efis: [
		cb.initNode("efis1",	1,	"BOOL"),
		cb.initNode("efis2",	1,	"BOOL"),
	],
	ldg_lt:	cb.initNode("landing-light",	1,	"BOOL"),
	fuel_pump:	cb.initNode("fuel-pump",	1,	"BOOL"),
	accy:		cb.initNode("accy",		1,	"BOOL"), # TODO what is this?
	xpdr:		cb.initNode("xpdr",		1,	"BOOL"),
	audio:	cb.initNode("audio",		1,	"BOOL"),
	flaps:	cb.initNode("flaps",		1,	"BOOL"),
	trim:		cb.initNode("trim",		1,	"BOOL"),
};

# Estimate electrical load:
# Landing Light: 	(based on https://aeroleds.com/products/sunspot-36lx-landing/): 45 Watts
# Navigation Lights and Strobe Lights based on https://aeroleds.com/wp-content/uploads/2018/02/0008-0006-Linear-Pulsar-NS-Series-Installation.pdf:
#	Navigation:		8.4 Watts (steady)			-> *2
#	Strobe:		70 Watts (during flash, 0.2 secs)	-> *2
#				11.2 Watts (average)			-> *2
# 	Fuel Pump: 		50 Watts (complete guess)
var consumers_main = { 
	# consumer name:  [  power (watts),  switch node,   circuit breaker node, output node ],
	landing_light:	[ 45, 	switches.landing, circuit_breakers.ldg_lt,	OutPuts.initNode("landing-light", 0.0, "DOUBLE")	],
	strobe_lights:	[ 22.4, 	switches.strobe, 	circuit_breakers.strobe_lts,	OutPuts.initNode("strobe-lights", 0.0, "DOUBLE")	],
	nav_lights:		[ 16.8, 	switches.nav,   	circuit_breakers.nav_lts,	OutPuts.initNode("nav-lights", 0.0, "DOUBLE")		],
	fuel_pump:		[ 50.0, 	switches.boost, 	circuit_breakers.fuel_pump,	OutPuts.initNode("fuel-pump", 0.0, "DOUBLE")		],
	# Starter has about 900W power according to https://www.ulforum.de/ultraleicht/forum/2_technik-und-flugzeuge/2466_anlasser-rotax-912-uls/seite-4.html
	starter:		[ 900.0,	switches.starter, nil,				 	OutPuts.initNode("starter", 0.0, "DOUBLE")		],
	
	trim:			[ 1.0,	nil,			circuit_breakers.trim,		OutPuts.initNode("trim",    0.0, "DOUBLE")		],
};

var avionics = {
	# consumer name:  [  power (watts),  switch node,   circuit breaker node, output node ],
	# Apollo SL40: 4W (receiving), 28W (transmit) ref. http://rv7-a.com/manuals/SL40Comm_InstallationManual.pdf p. 15 (printed page number)
	comm:			[ 4, serviceable.comm, circuit_breakers.radio[0],	OutPuts.initNode("comm", 0.0, "DOUBLE") ],
	# Garmin GTX-327: 15W (typical), 22W (max) ref. http://www.expaircraft.com/PDF/GTX327-IM.pdf p. 2 (printed page number)
	transponder:	[ 15, serviceable.xpdr, circuit_breakers.xpdr,		OutPuts.initNode("transponder", 0.0, "DOUBLE") ],
	# Dynon EFIS D-180: 14W (typical), 19W (max) ref. https://dynonavionics.com/includes/guides/FlightDEK-D180_Installation_Guide_Rev_H.pdf p. 9-38 (printed page number)
	efis:			[ 14, nil, circuit_breakers.efis[0],			OutPuts.initNode("efis", 0.0, "DOUBLE") ],
};

var strobe = aircraft.light.new("/sim/model/lights/strobe-lights", [0.2, 1.25], "/systems/electrical/outputs/strobe-lights");


var Battery = {
	new : func {
		m = { parents : [Battery] };
		m.ideal_volts = arg[0];
		m.ideal_amps = arg[1];
		m.amp_hours = arg[2];
		m.charge_percent = arg[3];
		m.charge_amps = arg[4];
		return m;
	},
	apply_load : func {
		var amphrs_used = arg[0] * (arg[1] / 3600.0);
		var percent_used = amphrs_used / me.amp_hours;
		me.charge_percent -= percent_used;
		if ( me.charge_percent < 0.0 ) {
			me.charge_percent = 0.0;
		} elsif ( me.charge_percent > 1.0 ) {
			me.charge_percent = 1.0;
		}
		return me.amp_hours * me.charge_percent;
	},
	get_output_volts : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_volts * factor;
	},
	get_output_amps : func {
		var x = 1.0 - me.charge_percent;
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},	
	recharge : func {
		me.charge_percent = 1.0;
	},
};

# var alternator = Alternator.new("rpm-source",rpm_threshold,volts,amps);

var Alternator = {
    new : func {
        m = { parents : [Alternator] };
        m.rpm_source =  props.globals.getNode(arg[0],1);
        m.rpm_threshold = arg[1];
        m.ideal_volts = arg[2];
        m.ideal_amps = arg[3];
        return m;
    },

    apply_load : func( amps, dt) {
        var factor = me.rpm_source.getValue() / me.rpm_threshold;
        if ( factor > 1.0 ){
            factor = 1.0;
        }
        var available_amps = me.ideal_amps * factor;
        return available_amps - amps;
    },

    get_output_volts : func {
        var factor = me.rpm_source.getValue() / me.rpm_threshold;
        if ( factor > 1.0 ) {
            factor = 1.0;
            }
        return me.ideal_volts * factor;
    },

    get_output_amps : func {
        var factor = me.rpm_source.getValue() / me.rpm_threshold;
        if ( factor > 1.0 ) {
            factor = 1.0;
        }
        return me.ideal_amps * factor;
    }
};

var battery = Battery.new(12.0, 20.0, 22.0, 1.0, 12.0);
var alternator = Alternator.new("/engines/engine[0]/rpm", 250.0, 14.0, 40.0);

setlistener("/sim/signals/fdm-initialized", func {
	electrical_timer.start();
});

var bus_volts = 0.0;
var load = 0.0;

var update_electrical = func {
	var dt = delta_t.getDoubleValue();
	var AltVolts = alternator.get_output_volts();
	var AltAmps = alternator.get_output_amps();
	var BatVolts = battery.get_output_volts();
	var BatAmps = battery.get_output_amps();
	var power_source = nil;

	if( serviceable.electric.getBoolValue() ){									# Electrical System is serviceable
		if ( switches.alt.getBoolValue() and (AltAmps > BatAmps)){
			bus_volts = AltVolts;
			power_source = "alternator";
			battery.apply_load(-battery.charge_amps, dt);			# Charge the battery
		}elsif ( switches.bat.getBoolValue()){
			bus_volts = BatVolts;
			power_source = "battery";
			battery.apply_load(load, dt);	# Load in ampere
		}else {
			bus_volts = 0.0;
		}
	} else {									# Electrical System not serviceable
		bus_volts = 0.0;
	}

	load = 0.0;
	load += electrical_bus(bus_volts);
	load += avionics_bus(bus_volts);
	
	eng_amp.setDoubleValue(bus_volts);		# TODO what does this affect?

	var bus_amps = 0.0;

	if (bus_volts > 1.0){
		if (power_source == "battery"){
			bus_amps = BatAmps-load;
		} else {
			bus_amps = battery.charge_amps;
		}
	}

	Amps.setValue(bus_amps);
	Volts.setValue(bus_volts);
}

var electrical_bus = func( bus_volts ){
	var load = 0.0;

	foreach( var key; keys( consumers_main ) ){
		if( ( consumers_main[key][1] == nil or consumers_main[key][1].getBoolValue() ) and ( consumers_main[key][2] == nil or consumers_main[key][2].getBoolValue() ) ){
			consumers_main[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( consumers_main[key][0] / bus_volts );
			}
		} else {
			consumers_main[key][3].setDoubleValue( 0.0 );
		}
	}

	return load;
}

var avionics_bus = func(bv) {
	
	var load = 0.0;
	var bus_volts = 0.0;
	
	if( switches.radio_master.getBoolValue() ){
		bus_volts = bv;
	}
	
	foreach( var key; keys( avionics ) ){
		if( ( avionics[key][1] == nil or avionics[key][1].getBoolValue() ) and ( avionics[key][2] == nil or avionics[key][2].getBoolValue() ) ){
			avionics[key][3].setDoubleValue( bus_volts );
			if( bus_volts != 0 ){
				load += ( avionics[key][0] / bus_volts );
			}
		} else {
			avionics[key][3].setDoubleValue( 0.0 );
		}
	}
	
	return load;
}

var electrical_timer = maketimer( 0.0, update_electrical );
electrical_timer.simulatedTime = 1;
