##
# Procedural model of a Glasfluegel Standard Libelle electrical system
# Based on C172P electrical system.

# Initialize properties
var com_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var com_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);
var vario_vol = props.globals.getNode("/instrumentation/ilec-sc7/volume", 1);
var vario_aud = props.globals.getNode("/instrumentation/ilec-sc7/audio", 1);
var vario_read = props.globals.getNode("/instrumentation/ilec-sc7/te-reading-mps", 1);
var tc_spin 	= props.globals.getNode("/instrumentation/turn-indicator/spin", 1);

var switch_prop = props.globals.getNode("/controls/electric", 1);

var switches = {
	master:	switch_prop.initNode("battery-switch",	0,	"BOOL"),
};

var cb_prop = props.globals.getNode("/controls/circuit-breakers", 1);

var circuit_breakers = {
	radio:	cb_prop.initNode("radio",	1,	"BOOL"),
	vario:	cb_prop.initNode("vario",	1,	"BOOL"),
	turn:		cb_prop.initNode("turn",	1,	"BOOL"),
};

var output_prop = props.globals.getNode("/systems/electrical/outputs", 1);

var outputs = {
	comm:		output_prop.initNode("comm[0]",	0.0,	"DOUBLE"),
	vario:	output_prop.initNode("ilec-sc7",	0.0,	"DOUBLE"),
	flarm:	output_prop.initNode("flarm",		0.0,	"DOUBLE"),
	tc:		output_prop.initNode("turn-coordinator",	0.0, "DOUBLE"),
};

var serviceable = props.globals.initNode("/systems/electrical/serviceable",	1,	"BOOL");

var amps = props.globals.initNode("/systems/electrical/amps",	0.0,	"DOUBLE");
var volts = props.globals.initNode("/systems/electrical/volts",	0.0,	"DOUBLE");

##
# Initialize internal values
#

var vbus_volts = 0.0;

var ammeter_ave = 0.0;
##
# Battery model class.
#

var BatteryClass = {};

BatteryClass.new = func {
	var obj = { parents : [BatteryClass],
		ideal_volts : 12.0,
		ideal_amps : 9.0,
		amp_hours : 9.0,
		charge_percent : props.globals.initNode("/systems/electrical/battery-charge-percent", 1.0, "DOUBLE"),
		charge_amps : 7.0 };
		return obj;
}

##
# Passing in positive amps means the battery will be discharged.
# Negative amps indicates a battery charge.
#

BatteryClass.apply_load = func (amps, dt) {
	var old_charge_percent = me.charge_percent.getDoubleValue();
	
	if (getprop("/sim/freeze/replay-state"))
		return me.amp_hours * old_charge_percent;
	
	var amphrs_used = amps * dt / 3600.0;
	var percent_used = amphrs_used / me.amp_hours;
	
	var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));
	
	if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
		gui.popupTip("Warning: Low battery! Recharge battery!", 10);
	
	me.charge_percent.setDoubleValue( new_charge_percent );
	
	return me.amp_hours * new_charge_percent;
}

##
# Return output volts based on percent charged.  Currently based on a simple
# polynomial percent charge vs. volts function.
#

BatteryClass.get_output_volts = func {
	var x = 1.0 - me.charge_percent.getDoubleValue();
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_volts * factor;
}


##
# Return output amps available.  This function is totally wrong and should be
# fixed at some point with a more sensible function based on charge percent.
# There is probably some physical limits to the number of instantaneous amps
# a battery can produce (cold cranking amps?)
#

BatteryClass.get_output_amps = func {
	var x = 1.0 - me.charge_percent.getDoubleValue();
	var tmp = -(3.0 * x - 1.0);
	var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
	return me.ideal_amps * factor;
}

##
# Set the current charge instantly to 100 %.
#

BatteryClass.reset_to_full_charge = func {
	me.apply_load(-(1.0 - me.charge_percent.getDoubleValue() ) * me.amp_hours, 3600);
}


var battery = BatteryClass.new();

var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach( var key; keys( circuit_breakers ) ){
		circuit_breakers[ key ].setBoolValue( 1 );
	}
}

var recharge_battery = func {
	# Charge battery to 100 %
	battery.reset_to_full_charge();
}
##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_virtual_bus(dt);
	}
};

##
# Model the system of relays and connections that join the battery,master/alt switches.
#

var update_virtual_bus = func (dt) {
	var load = 0.0;
	var battery_volts = 0.0;
	if ( serviceable.getBoolValue() ) {
		battery_volts = battery.get_output_volts();
	}
	
	# switch state
	var master_bat = switches.master.getBoolValue();
	
	
	# determine power source
	if (master_bat){
		var bus_volts = battery_volts;
		var power_source = battery;
	}else{
		var bus_volts = 0.0;
	}
	#print( "virtual bus volts = ", bus_volts );
	
	# bus network (1. these must be called in the right order, 2. the
	# bus routine itself determins where it draws power from.)
	load += electrical_bus_1();
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		bus_volts = 0;
	}
	
	# system loads and ammeter gauge
	var ammeter = 0.0;
	if ( bus_volts > 1.0 ) {
		# ammeter gauge
		if ( power_source == "battery" ) {
			ammeter = -load;
		} else {
			ammeter = battery.charge_amps;
		}
	}
	# print( "ammeter = ", ammeter );
	
	# charge/discharge the battery
	battery.apply_load( load, dt );
	
	# filter ammeter needle pos
	ammeter_ave = 0.8 * ammeter_ave + 0.2 * ammeter;
	
	# outputs
	amps.setDoubleValue( ammeter_ave );
	volts.setDoubleValue( bus_volts );
	if (bus_volts > 12)
		vbus_volts = bus_volts;
	else
		vbus_volts = 0.0;
	
	return load;
}

#Load sources:
#	com:			https://www.skyfox.com/becker-ar6201-022-vhf-am-sprechfunkgeraet-8-33.html
#	vario:		http://www.ilec-gmbh.com/ilec/manuals/SC7pd.pdf
#	flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
#	flarm display:	https://www.air-store.eu/Display-V3-FLARM
#	turn:			https://www.airteam.eu/de/p/falcon-tb02e-2-1 (not the same but similar)

var electrical_bus_1 = func() {
	var bus_volts = vbus_volts;
	var load = 0.0;
	
	if(bus_volts > 9){
		
		# Vario
		if( circuit_breakers.vario.getBoolValue() ){
			outputs.vario.setDoubleValue( bus_volts );
			#Energy consumption:	25mA (medium volume) 60mA (max volume) -> guess: at 12V
			#			guess: base consumption 5mA (no volume)
			load += 0.06 / bus_volts;
			if(vario_aud.getValue() == 2 or (vario_aud.getValue() == 1 and vario_read.getValue() > 0)){
				load += (vario_vol.getValue()*0.66) / bus_volts;
			}
		}else{
			outputs.vario.setDoubleValue( 0.0 );
		}
		
		
		# Radio
		if( circuit_breakers.radio.getBoolValue() ){
			outputs.comm.setDoubleValue( bus_volts );
			if(com_ptt.getBoolValue() and com_start.getValue()==1){
				load += 19.2 / bus_volts;
			}else{
				load += 1.02*com_start.getValue() / bus_volts;
			}
		}else{
			outputs.comm.setDoubleValue( 0.0 );
		}

		#Turn Coordinator
		#Energy Consumption:
		#	starting ~9.9 / volts (approx)
		#	running ~7.8 / volts (approx)
		if( circuit_breakers.turn.getBoolValue() ) {
			outputs.tc.setDoubleValue( bus_volts );
			if( tc_spin.getValue() > 0.99 ){
				load += 7.8 / bus_volts;
			}else{
				load += 9.9 / bus_volts;
			}
		} else {
			outputs.tc.setDoubleValue( 0.0 );
		}
		
		if( bus_volts > 9 ){
			outputs.flarm.setDoubleValue( bus_volts );
			load += 0.66 / bus_volts; #FLARM
			load += 0.12 / bus_volts; #FLARM display
		}else{
			outputs.flarm.setDoubleValue( 0.0 );
		}
	}else{
		outputs.vario.setDoubleValue( 0.0 );
		outputs.comm.setDoubleValue( 0.0 );
		outputs.flarm.setDoubleValue( 0.0 );
	}
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery.reset_to_full_charge();
	};
	
	system_updater.enable();
	print("Electrical system initialized");
});
