#	TinyTach TT2A Simulation
#	for FlightGear
#	by Bea Wolf
#	started 03/2024

#	Released under the terms of the GNU General Public License
#	version 2 or, at your option, any later version

#	References:
#		https://www.tinytach.com/gasoline-tinytach
#		https://www.tinytach.com/images/products/TT2A-standard-gas-tinytach.jpg
#		https://www.tinytach.com/pdf/Tiny2AInstructions11263.pdf
#		https://www.youtube.com/watch?v=usfmhpF6QB0

#	Reset Routine:
#		1. Change to JOB TIME display (assumed, TODO reference needed)
#		2. Press SELECT until RESET flag appears
#		3. Release SELECT -> JOB TIME is reset to 0

var time = props.globals.getNode("/sim/time/elapsed-sec", 1);

var tt2a = {
	new:	func( i ){
		var m = {
			parents: [ tt2a ],
		};
		m.mode = 0;	# Modes: 0 = Total Time, 1 = RPM, 2 = Job Timer
		m.digits = [
			props.globals.initNode("/instrumentation/tt2a["~ i ~"]/digit[0]", 0, "INT"),
			props.globals.initNode("/instrumentation/tt2a["~ i ~"]/digit[1]", 0, "INT"),
			props.globals.initNode("/instrumentation/tt2a["~ i ~"]/digit[2]", 0, "INT"),
			props.globals.initNode("/instrumentation/tt2a["~ i ~"]/digit[3]", 0, "INT"),
			props.globals.initNode("/instrumentation/tt2a["~ i ~"]/digit[4]", 0, "INT"),
		];
		m.flags = {
			tot:	props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/tot",   0, "BOOL"),
			job:	props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/job",   0, "BOOL"),
			reset:props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/reset", 0, "BOOL"),
			set:	props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/set",   0, "BOOL"),
			colon: [
				props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/colon[0]",   0, "BOOL"),
				props.globals.initNode("/instrumentation/tt2a["~ i ~"]/flags/colon[1]",   0, "BOOL"),
			],
		};
		m.total_time = props.globals.getNode("/instrumentation/tt2a["~ i ~"]/total-time-hr");
		m.job_time = props.globals.getNode("/instrumentation/tt2a["~ i ~"]/job-time-hr");
		m.rpm = props.globals.getNode("/engines/engine["~ i ~"]/rpm", 1);
		m.pressed_time = nil;
		
		return m;
	},
	update: func{
		var rpm = me.rpm.getDoubleValue();
		
		if( rpm > 1 ){
			me.mode = 1;
			me.set_flags();
			
			me.total_time.setDoubleValue( me.total_time.getDoubleValue() + ( 0.5 / 60 / 60 ) );
			me.job_time.setDoubleValue(   me.job_time.getDoubleValue()   + ( 0.5 / 60 / 60 ) );
		} elsif( me.mode == 1 ){
			me.mode = 0;
			me.set_flags();
		}
		
		
		if( me.mode == 0 ){
			
			var total_time = me.total_time.getDoubleValue();
			
			me.digits[0].setIntValue( int( total_time / 100 ) ); # 100 hours
			me.digits[1].setIntValue( int( total_time / 10 ) ); # 10 hours
			me.digits[2].setIntValue( math.mod( int( total_time ), 10 ) ); # 1 hour
			me.digits[3].setIntValue( 6 * math.mod( total_time, 1 ) ); # 10 minutes
			me.digits[4].setIntValue( 60 * math.mod( total_time, 1 ) ); # 1 minute
			
			me.flags.colon[1].setBoolValue( 1 );
			
		} elsif( me.mode == 1 ){
			# RPM Mode
			if( rpm >= 1000 ){
				rpm = math.round( rpm, 10 );
			}
			
			me.digits[0].setIntValue( int( rpm / 10000 ) );
			me.digits[1].setIntValue( math.mod( int( rpm / 1000 ), 10 ) );
			me.digits[2].setIntValue( math.mod( int( rpm / 100 ), 10 ) );
			me.digits[3].setIntValue( math.mod( int( rpm / 10 ), 10 ) );
			me.digits[4].setIntValue( math.mod( int( rpm / 1 ), 10 ) );
			
			me.flags.colon[1].setBoolValue( 0 );
		} elsif( me.mode == 2 ){
			var job_time = me.job_time.getDoubleValue();
			
			me.digits[0].setIntValue( int( job_time / 100 ) ); # 100 hours
			me.digits[1].setIntValue( int( job_time / 10 ) ); # 10 hours
			me.digits[2].setIntValue( math.mod( int( job_time ), 10 ) ); # 1 hour
			me.digits[3].setIntValue( 6 * math.mod( job_time, 1 ) ); # 10 minutes
			me.digits[4].setIntValue( 60 * math.mod( job_time, 1 ) ); # 1 minute
			
			me.flags.colon[1].setBoolValue( 1 );
			
			if( me.pressed_time != nil and ( time.getDoubleValue() - me.pressed_time ) > 1.5 ){
				# Set RESET Flag but don't reset time yet, this only happens after button is released
				me.flags.reset.setBoolValue( 1 );
				me.pressed_time = -1.0;	# identify button has been pressed long enough
			}
		}
	},
	set_flags: func{
		
		if( me.mode == 0 ){
			me.flags.tot.setBoolValue( 1 );
			me.flags.job.setBoolValue( 0 );
		} elsif( me.mode == 1 ){
			me.flags.tot.setBoolValue( 0 );
			me.flags.job.setBoolValue( 0 );
		} elsif( me.mode == 2 ){
			me.flags.tot.setBoolValue( 0 );
			me.flags.job.setBoolValue( 1 );
		}
		
	},
	btn_select: func( x ){
		if( x ){
			# pressed			
			if( me.mode == 2 ){
				# if JOB TIMER is displayed, store press time to check for "long press" = reset job time later
				me.pressed_time = time.getDoubleValue();
			}
		} else {
			# released
			if( me.pressed_time != nil and me.pressed_time == -1.0 ){
				# long press action
				me.job_time.setDoubleValue( 0.0 );
				me.flags.reset.setBoolValue( 0 );
			} else {
				# short press action
				if( me.mode == 0 ) me.mode = 2;
				elsif( me.mode == 2 ) me.mode = 0;
				me.set_flags();
			}
			
			me.pressed_time = nil;	# always reset press time to NIL as pressed_time != nil is used to identify button still being pressed
		}
	},
};

var instr = [
	tt2a.new( 0 ),
	tt2a.new( 1 ),
];

instr[0].set_flags();
instr[1].set_flags();

var instr_timer = [
	maketimer( 0.5, instr[0], instr[0].update ),
	maketimer( 0.5, instr[1], instr[1].update ),
];

foreach( var el; instr_timer ){
	el.simulatedTime = 1;
	el.start();
}

# Bind Listener Properties
var select_btn = [
		props.globals.initNode("/instrumentation/tt2a[0]/select-btn", 0, "BOOL"),
		props.globals.initNode("/instrumentation/tt2a[1]/select-btn", 0, "BOOL"),
];

setlistener( select_btn[0], func{
	instr[0].btn_select( select_btn[0].getBoolValue() );
});
setlistener( select_btn[1], func{
	instr[1].btn_select( select_btn[1].getBoolValue() );
});
