##
# Electric aircraft electrical system model
#

#	Internals
var eng_volts = electrical.initNode( "engine-volts", 0.0, "DOUBLE" );
var eng_limit = electrical.initNode( "engine-limit", 0.0, "DOUBLE" );
var aux_batt_serv = electrical.getNode("components/aux-battery-serviceable", 1);
var lv = {
	serv: electrical.initNode( "lv-serviceable", 1, "BOOL" ),
	volts: electrical.initNode( "lv-volts", 0.0, "DOUBLE" ),
	amps: electrical.initNode(  "lv-amps", 0.0, "DOUBLE" ),
};
var charger = props.globals.initNode( "/controls/electric/charger", 0, "BOOL" );
var hv = {
	serv: electrical.initNode(  "hv-serviceable", 1, "BOOL" ),
	volts: electrical.initNode( "hv-volts", 0.0, "DOUBLE" ),
	amps: electrical.initNode(  "hv-amps", 0.0, "DOUBLE" ),
};
var dcdc_avail = 0;
var dcdc_load = 0;
var freeze = props.globals.getNode( "/sim/freeze/replay-state", 1 );

# Switches
var switches = {
	master:		props.globals.getNode("/controls/switches/master", 1),
	batt:		props.globals.getNode("/controls/switches/batt",    1),
	navac_lt:	props.globals.getNode("/controls/lighting/nav-ac-lights",	1),
};

var batt_fan = [
	props.globals.getNode("/fdm/jsbsim/systems/cooling/battery-fan-requested[0]"),
	props.globals.getNode("/fdm/jsbsim/systems/cooling/battery-fan-requested[1]"),
];

#	Outputs
var otpts = electrical.initNode( "outputs" );
var outputs = {
	socket: otpts.initNode( "socket", 0.0, "DOUBLE"),
	nav_ac_lights: otpts.initNode( "nav-ac-lights", 0.0, "DOUBLE"),
	nav_lights_norm: otpts.initNode( "nav-lights-norm", 0.0, "DOUBLE"),
	ac_lights_bool: otpts.initNode( "ac-lights-bool", 0.0, "BOOL"),
	comm:		otpts.initNode( "comm[0]",	0.0, "DOUBLE"),
	pfd:		otpts.initNode( "pfd",    	0.0, "DOUBLE"),
	transponder:	otpts.initNode( "transponder",	0.0, "DOUBLE"),
	instruments:	otpts.initNode( "instruments",	0.0, "DOUBLE"),
	avionics:	otpts.initNode( "avionics",	0.0, "DOUBLE"),
	epsi:		otpts.initNode( "epsi",		0.0, "DOUBLE"),
	arm:		otpts.initNode( "prop-arm",	0.0, "DOUBLE"),
	vario:		otpts.initNode( "S3",		0.0, "DOUBLE"),
	flarm:		otpts.initNode( "flarm",	0.0, "DOUBLE"),
	batt_fan: [
			otpts.initNode( "battery-fan[0]",0.0, "DOUBLE"),
			otpts.initNode( "battery-fan[1]",0.0, "DOUBLE"),
		],
	batt_heat: [
			otpts.initNode( "battery-heating[0]",0.0, "DOUBLE"),
			otpts.initNode( "battery-heating[1]",0.0, "DOUBLE"),
		],
};


setlistener( batt.present[0], func {
	if( batt.present[0].getBoolValue() ){
		batt_present[0] = 1;
	} else {
		batt_present[0] = 0;
	}
});
setlistener( batt.present[1], func {
	if( batt.present[1].getBoolValue() ){
		batt_present[1] = 1;
	} else {
		batt_present[1] = 0;
	}
});

var lighting = [
	props.globals.getNode("/rendering/scene/ambient/red", 1),
	props.globals.getNode("/rendering/scene/ambient/green", 1),
	props.globals.getNode("/rendering/scene/ambient/blue", 1),
];

var comm_ptt = props.globals.getNode("/instrumentation/comm[0]/ptt", 1);
var comm_start = props.globals.getNode("/instrumentation/comm[0]/start", 1);

var xpdr_start = props.globals.getNode("/instrumentation/transponder/start", 1);

var avrg_chrg	=	electrical.initNode("average-charge", 0.0, "DOUBLE");

var cb_path = props.globals.initNode("/controls/circuit-breakers");

var cb = {
	instr:		cb_path.initNode("instr", 	1, "BOOL"),
	inv:		cb_path.initNode("inv",		1, "BOOL"),	# Power Controller
	batt:		cb_path.initNode("batt",	1, "BOOL"),
	arm:		cb_path.initNode("arm", 	1, "BOOL"),	# Propeller Arm Actuator
	avionics:	cb_path.initNode("avionics", 	1, "BOOL"),
	pfd:		cb_path.initNode("pfd", 	1, "BOOL"),
	com:		cb_path.initNode("com", 	1, "BOOL"),
	xpdr:		cb_path.initNode("xpdr", 	1, "BOOL"),
	socket:		cb_path.initNode("socket", 	1, "BOOL"),
	prop_disp:	cb_path.initNode("prop-disp", 	1, "BOOL"),	# EPSI 430	
	navac:		cb_path.initNode("nav-ac", 	1, "BOOL"),
};

# Lights
var anti_coll_lights = aircraft.light.new( "/systems/electrical/outputs/ac-lights-norm", [0.05, 0.8], "/systems/electrical/outputs/ac-lights-bool" );

##
# Battery model class.
#

var BatteryClass = {
	new: func (volts, amps, amp_hrs, charge_amps, name, cb = nil) {
		var obj = { parents : [BatteryClass],
			ideal_volts : volts,
			ideal_amps : amps,
			amp_hours : amp_hrs,
			name : name,
			cb: cb,
			charge_amps : charge_amps,
			charge_percent : props.globals.initNode("/systems/electrical/battery-"~name~"/charge-percent", 0.0, "DOUBLE"),
			amps: props.globals.initNode("/systems/electrical/battery-"~name~"/amps", 0.0, "DOUBLE"),
			volts: props.globals.initNode("/systems/electrical/battery-"~name~"/volts", 0.0, "DOUBLE"),
			state: 1,
		};
		return obj;
	},
	apply_load: func (amps, dt) {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		
		me.amps.setDoubleValue( amps );
		
		var old_charge_percent = me.charge_percent.getDoubleValue();
		
		if ( freeze.getBoolValue() ){	return me.amp_hours * old_charge_percent;	}
		
		var amphrs_used = amps * dt / 3600.0;
		var percent_used = amphrs_used / me.amp_hours;
		
		var new_charge_percent = std.max(0.0, std.min(old_charge_percent - percent_used, 1.0));
		
		if (new_charge_percent < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Apply external power to recharge battery!", 10);
		me.charge_percent.setDoubleValue( new_charge_percent );
		return me.amp_hours * new_charge_percent;
	},
	get_output_volts: func {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		var x = 1.0 - me.charge_percent.getDoubleValue();
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		var output_volts = me.ideal_volts * factor;
		me.volts.setDoubleValue( output_volts);
		return output_volts;
	},
	get_output_amps: func {
		if( !me.state or ( me.cb != nil and !me.cb.getBoolValue() ) ) return 0.0;
		var x = 1.0 - me.charge_percent.getDoubleValue();
		var tmp = -(3.0 * x - 1.0);
		var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
		return me.ideal_amps * factor;
	},
	reset_to_full_charge: func {
		me.apply_load(-(1.0 - me.charge_percent.getDoubleValue() ) * me.amp_hours, 3600);
	},
};

var solar_active = -1;

var SolarClass = {
	new: func( max_power, index ){
		var obj = { parents: [SolarClass],
			max_power: max_power,
			index: index,
			power_output: props.globals.initNode("/systems/electrical/solar["~ index ~"]/power-output"),
		};
		return obj;
	},
	get_current_power: func() {
		if( solar_active ){
			var current_output = me.max_power * math.max( lighting[0].getDoubleValue() + lighting[1].getDoubleValue() + lighting[2].getDoubleValue() - 0.6, 0 ) / 1.5;
		} else {
			var current_output = 0.0;
		}
		me.power_output.setDoubleValue( current_output );
		return current_output;
	},
};

var battery_aux = BatteryClass.new(13.2, 15, 12.4, 7, "aux"); 	# AOI, p.7-32 EarthX 680C
										# https://earthxbatteries.com/wp-content/uploads/2012/10/ETX680C-Product-Spec.pdf
										# TODO: ideal amps? (CB rated at 35A, max. continuous for battery 100A, graph shows 6.2, 12.4 and 18.6 A discharge
var battery_front = BatteryClass.new(266, 50, 30, 30, "front", cb.batt);	# AOI p.7-15 and 7-17: 30 Ah (optional: +10 Ah = 40 Ah), voltages: min 195, nominal 266, max 302 V, max. charging 30A (optional: 40A)
									# TODO: ideal amps?
var battery_rear  = BatteryClass.new(266, 50, 30, 30, "rear",  cb.batt);

var solar_panel = SolarClass.new( 22, 0 );

var reset_battery_and_circuit_breakers = func {
	# Charge battery to 100 %
	battery_aux.reset_to_full_charge();
	battery_front.reset_to_full_charge();
	battery_rear.reset_to_full_charge();
	
	reset_circuit_breakers();
}

var reset_circuit_breakers = func {
	foreach( var i; keys(cb) ){
		cb[i].setBoolValue( 1 );
	}
}


##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_lv_bus(dt);
		update_hv_bus(dt);
		update_solar_bus(dt);
	}
};

var update_solar_bus = func( dt ){
	var bus_volts = battery_aux.get_output_volts();
	if( bus_volts < 13.4 and solar_active != 1 ){
		solar_active = 1;
	} elsif( bus_volts > 13.5 and solar_active != 0 ){
		solar_active = 0;
	}
	var solar_power = solar_panel.get_current_power( bus_volts );
	if( solar_power > 0.1 ){
		bus_volts = 14;
		battery_aux.apply_load( math.min( - solar_power / bus_volts, battery_aux.charge_amps ), dt );
	}
}


var update_lv_bus = func (dt) {
	var load = 0.0;
	var battery_aux_volts = 0.0;
	if ( lv.serv.getBoolValue() and aux_batt_serv.getBoolValue() ) {
		battery_aux_volts = battery_aux.get_output_volts();
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	if ( switches.master.getBoolValue() ) {	# Master switch, connects aux battery to main bus
		bus_volts = battery_aux_volts;
		power_source = "battery";
	}
	if ( switches.batt.getBoolValue() and dcdc_avail ){
		bus_volts = 14.0;
		power_source = "dcdc";
	}
	
	load += avionics_bus( bus_volts );
	load += engine_bus( bus_volts );
	
	# Socket Power
	outputs.socket.setDoubleValue( bus_volts );
	load += bus_volts / 57;
	
	
	# NAV/AC Lights Power
	if ( switches.navac_lt.getBoolValue() ) {
		outputs.nav_ac_lights.setDoubleValue( bus_volts );
		if( bus_volts > 10 ) {
			outputs.nav_lights_norm.setDoubleValue( 1.0 );
			outputs.ac_lights_bool.setBoolValue( 1 );
		} else {
			outputs.nav_lights_norm.setDoubleValue( 0.0 );
			outputs.ac_lights_bool.setBoolValue( 0 );
		}
		load += bus_volts / 28;
	} else {
		outputs.nav_ac_lights.setDoubleValue( 0.0 );
		outputs.nav_lights_norm.setDoubleValue( 0.0 );
		outputs.ac_lights_bool.setBoolValue( 0 );
	}
	
	# swtich the master breaker off if load is out of limits
	if ( load > 55 ) {
		bus_volts = 0;
	}
	
	# charge/discharge the battery
	if ( power_source == "battery" ) {
		battery_aux.apply_load( load, dt );
	} elsif ( bus_volts > battery_aux_volts ) {
		battery_aux.apply_load( -battery_aux.charge_amps, dt );
		dcdc_load = load + battery_aux.charge_amps;
	}
	
	# Output properties
	lv.amps.setDoubleValue( load );
	lv.volts.setDoubleValue( bus_volts );
}

var batt_error = [ 0, 0 ];
var batt_heating_status = [ -1, -1 ];

var update_hv_bus = func (dt) {
	var load = 0.0;
	var battery_front_volts = 0.0;
	var battery_rear_volts = 0.0;
	var external_volts = 0.0;
	
	if ( hv.serv.getBoolValue() ) {
		if( !batt_error[0] and batt_present[0] ){
			battery_front_volts = battery_front.get_output_volts();
			annunciator.batt_front_offline = 0;
		} else {
			annunciator.batt_front_offline = 1;
		}
		if( !batt_error[1] > 0 and batt_present[1] ){
			battery_rear_volts = battery_rear.get_output_volts();
			annunciator.batt_rear_offline = 0;
		} else {
			annunciator.batt_rear_offline = 1;
		}
	}
	
	# determine power source
	var bus_volts = 0.0;
	var power_source = nil;
	
	if( charger.getBoolValue() and batt.temp[0].getDoubleValue() < 45 and batt.temp[1].getDoubleValue() < 45 ){
		external_volts = 230;
	}
	
	if ( switches.batt.getBoolValue() ) {
		if(battery_front_volts > 0){
			bus_volts = battery_front_volts;
			power_source = "battery_front";
		}
		if(battery_rear_volts > 0){
			if(bus_volts > 0){
				bus_volts += battery_rear_volts;
				power_source = "both_battery_packs";
				batt.status[1].setIntValue( 2 );
			}else{
				bus_volts = battery_rear_volts;
				power_source = "battery_rear";
				batt.status[1].setIntValue( 2 );
			}
		} else {
			batt.status[1].setIntValue( 1 );
		}
	}
	if ( external_volts > bus_volts ) {
		bus_volts = external_volts;
		power_source = "external";
	};
	
	
	if( batt_fan[0].getDoubleValue() > 0.0 and bus_volts > 100 ){
		outputs.batt_fan[0].setDoubleValue( bus_volts );
		load += 6 / bus_volts;
	} else {
		outputs.batt_fan[0].setDoubleValue( 0.0 );
	}
	
	if( batt_fan[1].getDoubleValue() > 0.0 and bus_volts > 100 ){
		outputs.batt_fan[1].setDoubleValue( bus_volts );
		load += 6 / bus_volts;
	} else {
		outputs.batt_fan[1].setDoubleValue( 0.0 );
	}
	
	if( batt.temp[0].getDoubleValue() < 15 and batt_heating_status[0] != 1 ) {
		batt_heating_status[0] = 1;
		annunciator.triggers[7].trigger();
	} elsif( batt.temp[0].getDoubleValue() > 17 and batt_heating_status[0] != 0 ){
		batt_heating_status[0] = 0;
	}
	
	if( batt.temp[1].getDoubleValue() < 15 and batt_heating_status[1] != 1 ) {
		batt_heating_status[1] = 1;
		if( !batt_heating_status[0] ) annunciator.triggers[7].trigger();
	} elsif( batt.temp[1].getDoubleValue() > 17 and batt_heating_status[1] != 0 ){
		batt_heating_status[1] = 0;
		if( !batt_heating_status[0] ) annunciator.triggers[7].reset();
	}
	
	if( batt_heating_status[0] and bus_volts > 100 ){
		outputs.batt_heat[0].setDoubleValue( bus_volts );
		load += 6 / bus_volts;
	} else {
		outputs.batt_heat[0].setDoubleValue( 0.0 );
	}
	if( batt_heating_status[1] and bus_volts > 100  ){
		outputs.batt_heat[1].setDoubleValue( bus_volts );
		load += 6 / bus_volts;
	} else {
		outputs.batt_heat[1].setDoubleValue( 0.0 );
	}
		
	if ( power_source == "both_battery_packs" ) {
		eng_limit.setDoubleValue( 1.0 );
	} else {
		eng_limit.setDoubleValue( 0.5833 );
	}
	
	#	The Power Controller H300C converts HV DC power to AC power for the engine
	if( cb.inv.getDoubleValue() and engine.sys_en.getBoolValue() and bus_volts > 0 and engine.prop_pos.getDoubleValue() > 0.99 ){
		if( engine.drive_en != 1 ){
			engine.drive_en = 1;
			annunciator.triggers[8].trigger();
		}
		
		eng_volts.setDoubleValue( bus_volts );
		
		var eng_power = engine.power_hp.getDoubleValue() * 745.7 * 1.1236 * 0.3; #hp to watts * ( 1 / motor efficiency) * calibration factor to achieve 1h endurance at 20kW (useful capacity 20kWh)
		
		if( engine.rpm.getDoubleValue() > 50 ){
			load += eng_power / bus_volts;   #P=U*I -> I=P/U
		}
	} else {
		eng_volts.setDoubleValue( 0.0 );
		
		if( engine.drive_en != 0 ){
			engine.drive_en = 0;
		}
	}
	
	#	The DC/DC converter reduces voltage to 14V to power the 14V buses and recharge the auxiliary battery
	if( bus_volts > 250 and dcdc_serv.getBoolValue() ){
		dcdc_avail = 1;
		load += dcdc_load * 14 / bus_volts;
	} else {
		dcdc_avail = 0;
	}
	
	# swtich the master breaker off if load is out of limits
	if ( load > 150 ) {
		bus_volts = 0;
	}
	
	if ( power_source == "both_battery_packs" ){
		batt.status[0].setIntValue( 2 );
		batt.status[1].setIntValue( 2 );
	} elsif( power_source == "battery_front" ){
		batt.status[0].setIntValue( 2 );
		if( batt_error[1] ){
			batt.status[1].setIntValue( 0 );
		} else {
			batt.status[1].setIntValue( 1 );
		}
	} elsif( power_source == "battery_rear" ){
		if( batt_error[0] ){
			batt.status[0].setIntValue( 0 );
		} else {
			batt.status[0].setIntValue( 1 );
		}
		batt.status[1].setIntValue( 2 );
	} else {
		if( batt_error[0] ){
			batt.status[0].setIntValue( 0 );
		} else {
			batt.status[0].setIntValue( 1 );
		}
		if( batt_error[1] ){
			batt.status[1].setIntValue( 0 );
		} else {
			batt.status[1].setIntValue( 1 );
		}
	}
		
	
	
	# charge/discharge the battery
	if ( power_source == "battery_front" ) {
		battery_front.apply_load( load, dt );
		battery_rear.apply_load( 0.0, dt );
	}else if ( power_source == "battery_rear" ) {
		battery_front.apply_load( 0.0, dt );
		battery_rear.apply_load( load, dt );
	}else if (power_source == "both_battery_packs" ) {
		battery_front.apply_load( 0.5*load, dt );
		battery_rear.apply_load( 0.5*load, dt );
	} else {
		battery_front.apply_load( 0.0, dt );
		battery_rear.apply_load( 0.0, dt );
	}
	
	avrg_chrg.setDoubleValue( ( battery_front.charge_percent.getDoubleValue() + battery_rear.charge_percent.getDoubleValue() ) / 2 );
	
	# outputs
	hv.amps.setDoubleValue( load );
	hv.volts.setDoubleValue( bus_volts );
}

#	Engine Bus: part of the 14V low voltage systems
#
#	Consumers:
#		
var engine_bus = func( bv ) {
	var bus_volts = bv;
	var load = 0.0;	
	
	# Engine Information System Power
	if ( cb.prop_disp.getBoolValue() ) {
		outputs.epsi.setDoubleValue( bus_volts );
	} else {
		outputs.epsi.setDoubleValue( 0.0);
	}
	
	# Propeller Arm Motor
	if( cb.arm.getBoolValue() ){
		outputs.arm.setDoubleValue( bus_volts );
		if( bus_volts > 9 ){
			engine.prop_cmd_int.setBoolValue( engine.prop_cmd.getBoolValue() );
		}
	} else {
		outputs.arm.setDoubleValue( 0.0 );
	}
	
	
	return load;
}

var avionics_bus = func( bv ) {
	var bus_volts = bv;
	var load = 0.0;
	
	if( bus_volts > 0 ){
		
		#	COM: ATR833-II-OLED
		#		Ref.: https://www.funkeavionics.de/produkt/atr833-ii-oled/
		#			receiving: 220/120 mA (assume: first number for 11V, second for 30V)
		#			transmitting: 1,3/0,65 A (assume: first number for 11V, second for 30V)
		if ( cb.com.getBoolValue() ) {
			outputs.comm.setDoubleValue( bus_volts );
			if( comm_start.getDoubleValue() > 0 ){
				if( comm_ptt.getBoolValue() ){
					load += 14.3 / bus_volts;	# 1.3A * 11V
				} else {
					load += 2.42 / bus_volts	# 0.22A * 11V
				}
			}
		} else {
			outputs.comm.setDoubleValue( 0.0 );
		}
		
		#	XPDR: TRT800H OLED
		#		Ref.: https://www.funkeavionics.de/produkt/trt800h-oled/
		#			mean amperage: < 300 mA at 13.8V, assume: 250 mA
		if ( cb.xpdr.getBoolValue() ) {
			outputs.transponder.setDoubleValue( bus_volts );
			if( xpdr_start.getDoubleValue() > 0 ){
				load += 3.45 / bus_volts;
			}
		} else {
			outputs.transponder.setDoubleValue( 0.0 );
		}
		
		# Other Instruments Power
	#		flarm:		http://flarm.com/wp-content/uploads/man/FLARM_InstallationManual_D.pdf
	#		flarm display:	https://www.air-store.eu/Display-V3-FLARM
	#		vario:		http://www.lx-avionik.de/produkte/s3/
		if ( cb.instr.getBoolValue() ) {
			outputs.instruments.setDoubleValue( bus_volts );
			
			outputs.vario.setDoubleValue( bus_volts );
			load += 1.380 / bus_volts; # 95mA - 135mA depending on display brightness setting at 12V
			
			if( bus_volts > 8 ){
				outputs.flarm.setDoubleValue( bus_volts );
				load += 0.66 / bus_volts; #FLARM
				load += 0.12 / bus_volts; #FLARM display
			} else {
				outputs.flarm.setDoubleValue( 0.0 );
			}
		} else {
			outputs.instruments.setDoubleValue( 0.0 );
			outputs.vario.setDoubleValue( 0.0 );
			outputs.flarm.setDoubleValue( 0.0 );
		}
	} else {
		outputs.comm.setDoubleValue( 0.0 );
		outputs.transponder.setDoubleValue( 0.0 );
		outputs.instruments.setDoubleValue( 0.0 );
		outputs.vario.setDoubleValue( 0.0 );
		outputs.flarm.setDoubleValue( 0.0 );
	}
	
	# return cumulative load
	return load;
}

##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

var el_ls = setlistener("/sim/signals/fdm-initialized", func {
	
	# checking if battery should be automatically recharged
	if (!getprop("/systems/electrical/save-battery-charge")) {
		battery_aux.reset_to_full_charge();
		battery_front.reset_to_full_charge();
		battery_rear.reset_to_full_charge();
	};
	reset_circuit_breakers();
	
	engine.prop_cmd_int.setBoolValue( engine.prop_cmd.getBoolValue() );
	
	system_updater.enable();
	
	removelistener( el_ls );
	print("Electrical system initialized");
});
