# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gettext
import os

import gtk
import gobject

from gazpacho import cursor, util
from gazpacho.placeholder import Placeholder
from gazpacho.properties import prop_registry, BooleanType, CustomProperty, \
     PropType, TransparentProperty, IntType, StringType, \
     PropertyCustomEditor, PropertyCustomEditorWithDialog
from gazpacho.uieditor import UIEditor
from gazpacho.widget import Widget, load_widget_from_gtk_widget
from gazpacho.widgetregistry import widget_registry
from gazpacho.widgetadaptor import WidgetAdaptor

_ = gettext.gettext

# The root library is the name of the library Gazpacho will import to create
# the widgets. e.g import gtk
root_library = 'gtk'

# When loading the widget class from the catalog files we strip off this prefix
# for every widget
widget_prefix = 'Gtk'

# This is in reality GtkWidget/GtkButton/GtkMenuItem:tooltip
def glade_gtk_widget_condition(widget_adaptor):
    button_gtype = gobject.type_from_name('GtkButton')
    menuitem_gtype = gobject.type_from_name('GtkMenuItem')
    
    if gobject.type_is_a(widget_adaptor.type, button_gtype):
        return True
    elif gobject.type_is_a(widget_adaptor.type, menuitem_gtype):
        return True
    else:
        return False # XXX TODO check for no window widgets

def glade_gtk_table_set_n_common(app, table, value, for_rows):
    new_size = value
    p = for_rows and 'n-rows' or 'n-columns'
    old_size = table.get_property(p)
    if new_size == old_size:
        return

    if new_size < 1:
        return

    gwidget = Widget.from_widget(table)
    if new_size > old_size:
        if for_rows:
            table.resize(new_size, table.get_property('n-columns'))
            for i in range(table.get_property('n-columns')):
                for j in range(old_size, table.get_property('n-rows')):
                    table.attach(Placeholder(app), i, i+1, j, j+1)
        else:
            table.resize(table.get_property('n-rows'), new_size)
            for i in range(old_size, table.get_property('n-columns')):
                for j in range(table.get_property('n-rows')):
                    table.attach(Placeholder(app), i, i+1, j, j+1)
    else:
        # Remove from the bottom up
        l = table.get_children()
        l.reverse()
        for child in l:
            p = for_rows and 'top-attach' or 'left-attach'
            start = table.child_get_property(child, p)
            p = for_rows and 'bottom-attach' or 'right-attach'
            end = table.child_get_property(child, p)

            # We need to completely remove it
            if start >= new_size:
                table.remove(child)
                continue
            # If the widget spans beyond the new border, we should resize it to
            # fit on the new table
            if end > new_size:
                p = for_rows and 'bottom-attach' or 'right-attach'
                table.child_set_property(child, p, new_size)
        table.resize(for_rows and new_size or table.get_property('n-rows'),
                     for_rows and table.get_property('n-columns') or new_size)

        p = for_rows and 'n-columns' or 'n-rows'
                       

class TableNRowsProp(TransparentProperty, IntType):
    default = 3
    def set(self, value):
        app = self._project.get_app()
        glade_gtk_table_set_n_common(app, self._object, value, True)
        self._value = value
        
    def get(self):
        return self._value
prop_registry.override_property('GtkTable::n-rows', TableNRowsProp)

class TableNColumnsProp(TransparentProperty, IntType):
    default = 3
    def set(self, value):
        app = self._project.get_app()
        glade_gtk_table_set_n_common(app, self._object, value, False)
        self._value = value
        
    def get(self):
        return self._value
prop_registry.override_property('GtkTable::n-columns', TableNColumnsProp)

class TreeViewAdaptor(WidgetAdaptor):
    def pre_create(self, context, tree_view, interactive=True):
        model = gtk.ListStore(str) # dummy model
        tree_view.set_model(model)

    # While we don't support column editing on the treeview
    # this does not make sense
##     renderer = gtk.CellRendererText()
##     column = gtk.TreeViewColumn('Column 1', renderer, text=0)
##     tree_view.append_column(column)

##     column = gtk.TreeViewColumn('Column 2', renderer, text=0)
##     tree_view.append_column(column)

    def fill_empty(self, context, widget):
        pass

class ComboBoxAdaptor(WidgetAdaptor):
    def pre_create(self, context, combo, interactive=True):
        model = gtk.ListStore(gobject.TYPE_STRING)
        combo.set_model(model)

    def fill_empty(self, context, widget):
        pass

class ContainerAdaptor(WidgetAdaptor):
    def replace_child(self, context, current, new, container):
        if current is None:
            container.add(new)
            return
        
        if current.parent != container:
            return

        props = {}
        for pspec in gtk.container_class_list_child_properties(type(container)):
            props[pspec.name] = container.child_get_property(current, pspec.name)

        container.remove(current)
        container.add(new)
            
        for name, value in props.items():
            container.child_set_property(new, name, value)

class WindowAdaptor(ContainerAdaptor):
    def post_create(self, context, window, interactive=True):
        window.set_default_size(440, 250)

    def fill_empty(self, context, gtk_widget):
        gtk_widget.add(context.create_placeholder())

### Managing the custom editors for GtkImage
# TODO: this has to be redone since now editors need to be classes, not just
# functions
## def _image_icon_editor_add_widget(box, name, gtk_widget, expand):
##     box.pack_start(gtk_widget, expand=expand)
##     box.set_data(name, gtk_widget)

## def glade_gtk_image_icon_editor_create(context):

##     store = gtk.ListStore(str, str)
##     completion = gtk.EntryCompletion()
##     completion.set_model(store)

##     cell = gtk.CellRendererPixbuf()
##     completion.pack_start(cell, True)
##     completion.add_attribute(cell, 'stock-id', 0)

##     cell = gtk.CellRendererText()
##     completion.pack_start(cell, True)
##     completion.add_attribute(cell, 'text', 1)

##     completion.set_text_column(0)
##     completion.set_minimum_key_length(1)
    
##     _image_icon_create_stock_popup(completion)
    
##     entry = gtk.Entry()
##     entry.set_completion(completion)
    
##     fc_button = gtk.Button(_('...'))

##     hbox = gtk.HBox()
##     _image_icon_editor_add_widget(hbox, 'image-icon-entry', entry, True)
##     _image_icon_editor_add_widget(hbox, 'image-icon-file-chooser', fc_button,
##                                   False)

##     return hbox

## def _image_icon_create_stock_popup(completion):
##     model = completion.get_model()
    
##     for stock_id in gtk.stock_list_ids():
##         if stock_id == 'gtk-missing-image': continue
##         stock_item = gtk.stock_lookup(stock_id)
##         if not stock_item:
##             continue
##         model.append([stock_id, stock_item[1]])

## def _image_icon_entry_changed(entry, proxy):
##     if entry.get_text() in gtk.stock_list_ids():
##         proxy.set_property('stock', entry.get_text())

## def _image_icon_file_clicked(button, (proxy, image, entry)):
##     dialog = gtk.FileChooserDialog('Chooser', None,
##                                    gtk.FILE_CHOOSER_ACTION_OPEN,
##                                    buttons=(gtk.STOCK_CANCEL,
##                                             gtk.RESPONSE_CANCEL,
##                                             gtk.STOCK_OPEN,
##                                             gtk.RESPONSE_OK))

##     filter = gtk.FileFilter()
##     filter.set_name("Images")
##     filter.add_mime_type("image/png")
##     filter.add_mime_type("image/jpeg")
##     filter.add_mime_type("image/gif")
##     filter.add_mime_type("image/x-xpixmap")
##     filter.add_pattern("*.png")
##     filter.add_pattern("*.jpg")
##     filter.add_pattern("*.gif")
##     filter.add_pattern("*.xpm")

##     dialog.add_filter(filter)
    
##     response = dialog.run()
##     if response == gtk.RESPONSE_OK and dialog.get_filename():
##         entry.handler_block(entry.get_data('handler-id-changed'))
##         entry.set_text(os.path.basename(dialog.get_filename()))
##         entry.handler_unblock(entry.get_data('handler-id-changed'))
##         proxy.set_property('file', dialog.get_filename())
##         image.set_data('image-file-name', entry.get_text())

##     dialog.destroy()
##     return
 
## def _image_icon_reconnect(gtk_widget, signal, callback, userdata):
##     handler_id = gtk_widget.get_data('handler-id-' + signal)
##     if handler_id:
##         gtk_widget.disconnect(handler_id)

##     handler_id = gtk_widget.connect(signal, callback, userdata)
##     gtk_widget.set_data('handler-id-' + signal, handler_id)

## def glade_gtk_image_icon_editor_update(context, box, image, proxy):
##     entry = box.get_data('image-icon-entry')
##     toggle = box.get_data('image-icon-stock-chooser')
##     fc_button = box.get_data('image-icon-file-chooser')

##     stock_id = image.get_property('stock')
##     if stock_id and stock_id != 'gtk-missing-image':
##         text = stock_id
##     elif image.get_data('image-file-name'):
##         text = image.get_data('image-file-name')
##     else:
##         text = ''

##     entry.set_text(text)

##     _image_icon_reconnect(entry, 'changed', _image_icon_entry_changed, proxy)
##     _image_icon_reconnect(fc_button, 'clicked', _image_icon_file_clicked,
##                           (proxy, image, entry))

## def glade_gtk_image_icon_size_editor_create(context):
##     store = gtk.ListStore(str, int)

##     combo = gtk.ComboBox(store)
    
##     cell = gtk.CellRendererText()
##     combo.pack_start(cell, True)
##     combo.add_attribute(cell, 'text', 0)

##     return combo

## def _image_icon_size_setup_combo(editor, image):
##     model = editor.get_model()
##     model.clear()
    
##     stock_id = image.get_property('stock')
##     if not stock_id or stock_id == 'gtk-missing-image':
##         editor.set_sensitive(False)
##         return

##     icon_set = gtk.icon_factory_lookup_default(stock_id)
##     editor.set_sensitive(True)
    
##     icon_size = image.get_property('icon-size')
##     for size in icon_set.get_sizes():
##         iter = model.append([gtk.icon_size_get_name(size), size])

##         if size == icon_size:
##             editor.handler_block(editor.get_data('handler-id-changed'))
##             editor.set_active_iter(iter)
##             editor.handler_unblock(editor.get_data('handler-id-changed'))

## def _image_icon_size_notify(image, param_spec, (editor, proxy)):
##     _image_icon_size_setup_combo(editor, image)

## def _image_icon_size_changed(editor, proxy):
##     iter = editor.get_active_iter()
##     if iter:
##         model = editor.get_model()
##         proxy.set_value(model[iter][1])

## def glade_gtk_image_icon_size_editor_update(context, editor, image, proxy):
##     handler_id = image.get_data('image-notify-id')
##     if handler_id:
##         image.disconnect(handler_id)

##     _image_icon_reconnect(editor, 'changed', _image_icon_size_changed, proxy)

##     _image_icon_size_setup_combo(editor, image)
    
##     handler_id = image.connect('notify', _image_icon_size_notify,
##                                (editor, proxy))
##     image.set_data('image-notify-id', handler_id)
##     return


class DialogAdaptor(ContainerAdaptor):
    def child_property_applies(self, context, ancestor, gtk_widget,
                               property_id):
        if property_id == 'response-id' and \
           isinstance(gtk_widget.parent, gtk.HButtonBox) and \
           isinstance(gtk_widget.parent.parent, gtk.VBox) and \
           gtk_widget.parent.parent.parent == ancestor:
            return True
        elif gtk_widget.parent == ancestor:
            return True

        return False
    
    def post_create(self, context, dialog, interactive=True):
        gwidget = Widget.from_widget(dialog)
        if not gwidget: return
        
        # create the GladeWidgets for internal children
        self._setup_internal_children(gwidget)
                
        dialog.action_area.pack_start(context.create_placeholder())
        dialog.action_area.pack_start(context.create_placeholder())

        dialog.vbox.pack_start(context.create_placeholder())
        
        # set a reasonable default size for a dialog
        dialog.set_default_size(320, 260)

    def _setup_internal_children(self, gwidget):
        child_class = widget_registry.get_by_name('GtkVBox')
        vbox_widget = Widget(child_class, gwidget.project)
        vbox_widget.setup_internal_widget(gwidget.gtk_widget.vbox, 'vbox',
                                          gwidget.name or '')
        child_class = widget_registry.get_by_name('GtkHButtonBox')
        action_area_widget = Widget(child_class, gwidget.project)
        action_area_widget.setup_internal_widget(gwidget.gtk_widget.action_area,
                                                 'action_area',
                                                 gwidget.name or '')
        
class MessageDialogAdaptor(DialogAdaptor):
    def post_create(self, context, dialog, interactive=True):
        dialog.set_default_size(400, 115)

class NotebookAdaptor(ContainerAdaptor):
    def replace_child(self, context, current, new, container):
        page_num = container.page_num(current)
        if page_num == -1:
            return

        page = container.get_nth_page(page_num)
        label = container.get_tab_label(current)

        container.remove_page(page_num)
        container.insert_page(new, label, page_num)

        new.show()
        container.set_current_page(page_num)

class PanedAdaptor(ContainerAdaptor):
    def fill_empty(self, context, paned):
        paned.add1(context.create_placeholder())
        paned.add2(context.create_placeholder())
        # we want to set the position in the middle but
        # as the paned has not yet a parent it is not
        # realized we can't know its size
        gobject.idle_add(self._set_position, paned)

    def _set_position(self, paned):
        """This function get called until the paned is realized.
        Then it puts the bar in the middle.
        """
        if paned.flags() & gtk.REALIZED:
            alloc = paned.get_allocation()
            if isinstance(paned, gtk.VPaned):
                pos = alloc.height / 2
            else:
                pos = alloc.width / 2
            paned.set_position(pos)
            return False
        else:
            return True


# these widgets does not need a viewport when added to a scrolledwindow
scrollable_widgets = (gtk.TreeView, gtk.TextView, gtk.Viewport)

class ScrolledWindowAdaptor(ContainerAdaptor):

    def fill_empty(self, context, scrolledwindow):
        self._add_viewport(scrolledwindow, context.get_project())

    def _add_viewport(self, scrolledwindow, project):
        """ScrolledWindow should be empty before calling this method"""
        klass = widget_registry.get_by_name('GtkViewport')
        viewport = Widget(klass, project)
        viewport.create_gtk_widget(interactive=False)
        scrolledwindow.add(viewport.gtk_widget)
        project.add_widget(viewport.gtk_widget)
        viewport.gtk_widget.show_all()
        return viewport
        
    def replace_child(self, context, current, new, container):
        global scrollable_widgets
        
        if current is None:
            raise ValueError("Can not replace None widget")

        # we don't care about current since it is probably different from
        # our child because we always have a viewport when adding a widget
        # that is not scrollable
        child = container.get_child()
        container.remove(child)
        project = context.get_project()
        if child is not current:
            project.remove_widget(child)
        
        # luckily viewports or scrolledwindow does not have any packing
        # properties

        if isinstance(new, scrollable_widgets):
            container.add(new)
        elif isinstance(new, Placeholder):
            self._add_viewport(container, project)
        else:
            raise ValueError("You can only put scrollable widgets or "
                             "placeholders inside a ScrollableWindow: "
                             "%s" % new)

class ViewportAdaptor(ContainerAdaptor):

    def replace_child(self, context, current, new, container):
        """If the new child is a scrollable widget and our parent is a
        scrolled window, we remove ourselves"""
        global scrollable_widgets
        parent = container.get_parent()
        if (isinstance(parent, gtk.ScrolledWindow)
            and isinstance(new, scrollable_widgets)):
            project = context.get_project()
            project.remove_widget(container)
            parent.remove(container)
            parent.add(new)
        else:
            ContainerAdaptor.replace_child(self, context, current, new,
                                           container)

    def fill_empty(self, context, gtk_widget):
        gtk_widget.add(context.create_placeholder())

class FixedAdaptor(ContainerAdaptor):
    def __init__(self):
        # save the last button click coordinate
        self.x = 0
        self.y = 0

    def post_create(self, context, fixed, interactive=True):
        fixed.connect('expose-event', self.on_expose_event)

    def replace_child(self, context, current, new, container):
        "'current' is always None for Fixed since it never has any placeholder"
        container.put(new, self.x, self.y)

    def button_release(self, context, fixed, event):
        # we remember the point where the user clicked
        toplevel = fixed.get_toplevel()
        if toplevel:
            self.x, self.y = toplevel.translate_coordinates(fixed,
                                                            int(event.x),
                                                            int(event.y))
        
        # if there is some widget selected in the palette it's time to add it
        project = context.get_project()
        if project._app.add_class:
            cm = context.get_command_manager()
            gwidget = Widget.from_widget(fixed)
            cm.create(project._app.add_class, None, project, gwidget)
        return False

    def motion_notify(self, context, fixed, event):
        app = context.get_project()._app
        cursor.Cursor.set_for_widget_adaptor(event.window, app.add_class)
        return False

    def on_expose_event(self, fixed, event):
        # draw a grid
        window = fixed.window
        bw = fixed.get_property('border-width')
        w, h = window.get_size()
        gc = fixed.style.dark_gc[gtk.STATE_NORMAL]
        for gridx in range(bw, w-bw, 10):
	  for gridy in range(bw, h-bw, 10):
	    window.draw_point(gc, gridx, gridy)

        return False
    
class BaseAttach:
    """Base class for LeftAttach, RightAttach, TopAttach and BottomAttach
    adaptors"""
    def _get_attach(self, child):
        """Returns the four attach packing properties in a tuple"""
        right = self.table.child_get_property(child, 'right-attach')
        left = self.table.child_get_property(child, 'left-attach')
        top = self.table.child_get_property(child, 'top-attach')
        bottom = self.table.child_get_property(child, 'bottom-attach')
        return (left, right, top, bottom)

    def _cell_empty(self, x, y):
        """Returns true if the cell at x, y is empty. Exclude child from the
        list of widgets to check"""
        empty = True
        for w in self.table.get_children():
            left, right, top, bottom = self._get_attach(w)
            if (left <= x and (x + 1) <= right 
                and top <= y and (y + 1) <= bottom):
                empty = False
                break

        return empty

    def _create_placeholder(self, context, x, y):
        """Puts a placeholder at cell (x, y)"""
        placeholder = context.create_placeholder()
        self.table.attach(placeholder, x, x+1, y, y+1)

    def _fill_with_placeholders(self, context, y_range, x_range):
        """Walk through the table creating placeholders in empty cells.
        Only iterate between x_range and y_range.
        Child is excluded in the computation to see if a cell is empty
        """
        for y in range(self.n_rows):
            for x in range(self.n_columns):
                if self._cell_empty(x, y):
                    self._create_placeholder(context, x, y)

    def _initialize(self, child, prop_name):
        """Common things all these adaptors need to do at the beginning"""
        self.table = child.get_parent()
        (self.left_attach,
         self.right_attach,
         self.top_attach,
         self.bottom_attach) = self._get_attach(child)

        self.n_columns = self.table.get_property('n-columns')
        self.n_rows = self.table.get_property('n-rows')

        self.prop_name = prop_name
        self.child = child
        self.gchild = Widget.from_widget(self.child)
        self.gprop = self.gchild.get_prop(self.prop_name)

    def _value_is_between(self, value, minimum, maximum):
        if value < minimum:
            self.gprop._value = minimum
            return False

        if value > maximum:
            self.gprop._value = maximum
            return False

        return True

    def _internal_set(self, context, value, minimum, maximum,
                      y_range, x_range):
        """Check if value is between minium and maximum and then remove or
        add placeholders depending if we are growing or shrinking.
        If we are shrinking check the cells in y_range, x_range to add
        placeholders
        """
        if not self._value_is_between(value, minimum, maximum):
            return

        placeholder = context.create_placeholder()

        # are we growing?
        if self._is_growing(value):
            # check if we need to remove some placeholder
            for ph in filter(lambda w: isinstance(w, type(placeholder)),
                             self.table.get_children()):
                lph, rph, tph, bph = self._get_attach(ph)
                if self._cover_placeholder(value, lph, rph, tph, bph):
                    self.table.remove(ph)

            self.table.child_set_property(self.child, self.prop_name, value)

        # are we shrinking? maybe we need to create placeholders
        elif self._is_shrinking(value):
            self.table.child_set_property(self.child, self.prop_name, value)
            self._fill_with_placeholders(context, y_range, x_range)


    # virtual methods that should be implemented by subclasses:
    def _is_growing(self, value):
        """Returns true if the child widget is growing"""

    def _is_shrinking(self, value):
        """Returns true if the child widget is shrinking"""

    def _cover_placeholder(self, value, left, right, top, bottom):
        """Return True if there is a placeholder in these coordinates"""
        
class LeftAttachAdaptor(BaseAttach):
    def _is_growing(self, value):
        return value < self.left_attach

    def _is_shrinking(self, value):
        return value > self.left_attach

    def _cover_placeholder(self, value, left, right, top, bottom):
        if value < right and self.left_attach > left:
            if top >= self.top_attach and bottom <= self.bottom_attach:
                return True
        return False

    def set(self, context, child, value):
        self._initialize(child, 'left-attach')
        self._internal_set(context, value, 0, self.right_attach - 1,
                           range(self.n_rows),
                           range(self.left_attach, value))

class RightAttachAdaptor(BaseAttach):
    def _is_growing(self, value):
        return value > self.right_attach

    def _is_shrinking(self, value):
        return value < self.right_attach

    def _cover_placeholder(self, value, left, right, top, bottom):
        if value > left and self.right_attach < right:
            if top >= self.top_attach and bottom <= self.bottom_attach:
                return True
        return False

    def set(self, context, child, value):
        self._initialize(child, 'right-attach')
        self._internal_set(context, value,
                           self.left_attach + 1, self.n_columns,
                           range(value, self.right_attach),
                           range(self.n_rows))

class BottomAttachAdaptor(BaseAttach):
    def _is_growing(self, value):
        return value > self.bottom_attach

    def _is_shrinking(self, value):
        return value < self.bottom_attach

    def _cover_placeholder(self, value, left, right, top, bottom):
        if value > top and self.bottom_attach < bottom:
            if left >= self.left_attach and right <= self.right_attach:
                return True                
        return False

    def set(self, context, child, value):
        self._initialize(child, 'bottom-attach')
        self._internal_set(context, value,
                           self.top_attach + 1, self.n_rows,
                           range(value, self.bottom_attach),
                           range(self.n_columns))

class TopAttachAdaptor(BaseAttach):
    def _is_growing(self, value):
        return value < self.top_attach

    def _is_shrinking(self, value):
        return value > self.top_attach

    def _cover_placeholder(self, value, left, right, top, bottom):
        if value < bottom and self.top_attach > top:
            if left >= self.left_attach and right <= self.right_attach:
                return True
        return False

    def set(self, context, child, value):
        self._initialize(child, 'top-attach')
        self._internal_set(context, value, 0, self.bottom_attach - 1,
                           range(self.n_columns),
                           range(self.top_attach, value))

class TableAdaptor(ContainerAdaptor):

    def pre_create(self, context, table, interactive=True):
        """ a GtkTable starts with a default size of 1x1, and setter/getter of
        rows/columns expect the GtkTable to hold this number of placeholders,
        so we should add it. """
        table.attach(context.create_placeholder(), 0, 1, 0, 1)

    def post_create(self, context, table, interactive=True):
        if not interactive:
            return
        gwidget = Widget.from_widget(table)
        property_rows = gwidget.get_prop('n-rows')
        property_cols = gwidget.get_prop('n-columns')
        dialog = gtk.Dialog(_('Create a table'), None,
                            gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT | \
                            gtk.DIALOG_NO_SEPARATOR,
                            (gtk.STOCK_OK, gtk.RESPONSE_ACCEPT))
        dialog.set_position(gtk.WIN_POS_MOUSE)
        dialog.set_default_response(gtk.RESPONSE_ACCEPT)
        
        label_rows = gtk.Label(_('Number of rows')+':')
        label_rows.set_alignment(0.0, 0.5)
        label_cols = gtk.Label(_('Number of columns')+':')
        label_cols.set_alignment(0.0, 0.5)
        
        spin_button_rows = gtk.SpinButton()
        spin_button_rows.set_increments(1, 5)
        spin_button_rows.set_range(1.0, 10000.0)
        spin_button_rows.set_numeric(False)
        spin_button_rows.set_value(3)
        spin_button_rows.set_property('activates-default', True)
        
        spin_button_cols = gtk.SpinButton()
        spin_button_cols.set_increments(1, 5)
        spin_button_cols.set_range(1.0, 10000.0)
        spin_button_cols.set_numeric(False)
        spin_button_cols.set_value(3)
        spin_button_cols.set_property('activates-default', True)
        
        table = gtk.Table(2, 2, True)
        table.set_col_spacings(4)
        table.set_border_width(12)
        
        table.attach(label_rows, 0, 1, 0, 1)
        table.attach(spin_button_rows, 1, 2, 0, 1)
        
        table.attach(label_cols, 0, 1, 1, 2)
        table.attach(spin_button_cols, 1, 2, 1, 2)
        
        dialog.vbox.pack_start(table)
        table.show_all()
        
        # even if the user destroys the dialog box, we retrieve the number and
        # we accept it.  I.e., this function never fails
        dialog.run()
        
        property_rows.set(spin_button_rows.get_value_as_int())
        property_cols.set(spin_button_cols.get_value_as_int())
        
        dialog.destroy()

class UIPropEditor(PropertyCustomEditorWithDialog):
    dialog_class = UIEditor
    button_text = _('Edit UI Definition...')

class UIAdaptor(TransparentProperty, StringType):
    """This represents a fake property to edit the ui definitions of
    menubars and toolbars.

    It does not store anything because this information is stored in
    the uimanager itself. It's just a way to put a button in the Gazpacho
    interface to call the UIEditor.
    """
    editor = UIPropEditor
    
class MenuBarUIAdapter(UIAdaptor):
    pass

prop_registry.override_property('GtkMenuBar::ui', MenuBarUIAdapter)

class ToolbarUIAdapter(UIAdaptor):
    pass

prop_registry.override_property('GtkToolbar::ui', ToolbarUIAdapter)

class CommonBarsAdaptor(ContainerAdaptor):

    def post_create(self, context, gtk_widget, ui_string):
        # create some default actions
        gwidget = Widget.from_widget(gtk_widget)
        project = gwidget.project
        project.uim.create_default_actions()

        project.uim.add_ui(gwidget, ui_string)
        new_gtk_widget = project.uim.get_gtk_widget(gwidget)

        # we need to replace gtk_widget with new_widget
        gwidget.setup_widget(new_gtk_widget)
        #gwidget.apply_properties()
    
    def save(self, context, gwidget):
        """This saver is needed to avoid saving the children of toolbars
        and menubars
        """
        gwidget.constructor = 'initial-state'

    def load(self, context, gtk_widget, blacklist):
        """This loader is special because of these features:
        - It does not load the children of the menubar/toolbar
        - Load the uimanager and put its content (action groups) into the
        project
        """
        
#         # we need to save the properties of this gtk_widget because otherwise
#         # when we got it from the uimanager it's gonna be another widget with
#         # different properties
#         props = {}
#         for prop in gobject.list_properties(gtk_widget):
#             if 1 or prop.flags != gobject.PARAM_READWRITE:
#                 continue
#             if propertyclass.get_type_from_spec(prop) is gobject.TYPE_OBJECT:
#                 continue
#             # FIXME: This need to use the values from the catalog.
#             # But it doesn't work right now, the property in
#             # klass.properties is always set to False.
#             if prop.name == 'parent' or prop.name == 'child':
#                 continue
#             props[prop.name] = gtk_widget.get_property(prop.name)

        project = context.get_project()

        old_name = gtk_widget.name
        gwidget = Widget.load(gtk_widget, project, blacklist)
        gwidget._name = gwidget.gtk_widget.name

        # change the gtk_widget for the one we get from the uimanager
        project.uim.load_widget(gwidget, old_name)

        #gwidget.load_properties()
        gwidget.load_signals()

        return gwidget
    
class MenuBarAdaptor(CommonBarsAdaptor):

    def post_create(self, context, menubar, interactive=True):
        gwidget = Widget.from_widget(menubar)
        # A None in this list means a separator
        names = [gwidget.name, _('FileMenu'), _('New'), _('Open'), _('Save'),
                 _('SaveAs'), _('Quit'), _('EditMenu'), _('Copy'), _('Cut'),
                 _('Paste')]
        tmp = []
        for name in names:
            tmp.extend([name, name])
        
        ui_string = """<menubar action="%s" name="%s">
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <separator/>
    <menuitem action="%s" name="%s"/>
  </menu>
  <menu action="%s" name="%s">
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
    <menuitem action="%s" name="%s"/>
  </menu>
</menubar>""" % tuple(tmp)
    
        super(MenuBarAdaptor, self).post_create(context, menubar, ui_string)

class ToolbarAdaptor(CommonBarsAdaptor):
    def post_create(self, context, toolbar, interactive=True):
        gwidget = Widget.from_widget(toolbar)

        names = [gwidget.name, _('New'), _('Open'), _('Save'),
                 _('Copy'), _('Cut'), _('Paste')]
        tmp = []
        for name in names:
            tmp.extend([name, name])
    
        ui_string = """<toolbar action="%s" name="%s">
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <separator/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
  <toolitem action="%s" name="%s"/>
</toolbar>""" % tuple(tmp)

        super(ToolbarAdaptor, self).post_create(context, toolbar, ui_string)

(COL_STOCK_ID,
 COL_STOCK_LABEL) = range(2)

# Adaptors for Button
class ButtonContentsEditor(PropertyCustomEditor):

    wide_editor = True
    
    def __init__(self):
        self.loading = False
        self.gwidget = None
        self.application_window = None
        self.input = self.create()

    def get_editor_widget(self):
        return self.input
    
    def _create_stock_list(self):
        "Create the list with the stock icons"
        liststore = gtk.ListStore(str, str)
        stocks = gtk.stock_list_ids()
        stocks.sort()
        for stock in stocks:
            info = gtk.stock_lookup(stock)
            if info is None:
                label = "* %s" % stock
            else:
                label = info[1]
            liststore.append((stock, label.replace('_', '')))

        treeview = gtk.TreeView(liststore)
        treeview.set_headers_visible(False)
        column = gtk.TreeViewColumn()
        cell1 = gtk.CellRendererPixbuf()
        column.pack_start(cell1)
        column.add_attribute(cell1, 'stock-id', COL_STOCK_ID)
        cell2 = gtk.CellRendererText()
        column.pack_start(cell2)
        column.add_attribute(cell2, 'text', COL_STOCK_LABEL)
        treeview.append_column(column)
        treeview.set_search_column(COL_STOCK_LABEL)
        treeview.set_enable_search(True)

        self.stock_list = treeview

        treeview.get_selection().connect('changed',
                                         self._on_stock_list_selection_changed)

        sw = gtk.ScrolledWindow()
        sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        sw.set_shadow_type(gtk.SHADOW_IN)
        sw.add(treeview)

        return sw

    def _create_check_entry_button(self, table, row, check_text, button_text,
                                   entry_handler, check_handler,
                                   button_handler):
        check = gtk.CheckButton(check_text)
        table.attach(check, 0, 1, row, row + 1,
                     xoptions=gtk.FILL, yoptions=gtk.FILL)
        hbox = gtk.HBox()
        entry = gtk.Entry()
        entry.connect('changed', entry_handler)
        hbox.pack_start(entry)
        button = gtk.Button(button_text)
        button.connect('clicked', button_handler)
        hbox.pack_start(button, False, False)
        table.attach(hbox, 1, 2, row, row + 1, xoptions=gtk.EXPAND|gtk.FILL,
                     yoptions=gtk.EXPAND|gtk.FILL)
        check.connect('toggled', check_handler)
        return check, entry, button
        
    def _create_label(self, table):
        "Create the widgets needed to set the label of a button"
        c, e, b = self._create_check_entry_button(table, 0, _('Label:'), '...',
                                                  self._on_entry_label_changed,
                                                  self._on_check_label_toggled,
                                                  self._on_button_label_clicked)
        self.check_label, self.entry_label, self.button_label = c, e, b


    def _create_image(self, table):
        "Create the widgets needed to set the image of a button"
        c, e, b = self._create_check_entry_button(table, 1, _('Image:'), '...',
                                                  self._on_entry_image_changed,
                                                  self._on_check_image_toggled,
                                                  self._on_button_image_clicked)
        self.check_image, self.entry_image, self.button_image = c, e, b


    def _create_position(self, table):
        """Create the widgets needed to set the combo with the position
        of the image"""
        label = gtk.Label(('Image position:'))
        label.set_sensitive(False)
        table.attach(label, 0, 1, 2, 3, xoptions=gtk.FILL, yoptions=gtk.FILL)

        liststore = gtk.ListStore(str, int)
        liststore.append((_('Left'), gtk.POS_LEFT))
        liststore.append((_('Right'), gtk.POS_RIGHT))
        liststore.append((_('Top'), gtk.POS_TOP))
        liststore.append((_('Bottom'), gtk.POS_BOTTOM))
        combo = gtk.ComboBox(liststore)
        renderer = gtk.CellRendererText()
        combo.pack_start(renderer)
        combo.add_attribute(renderer, 'text', 0)

        combo.set_sensitive(False)
        table.attach(combo, 1, 2, 2, 3, xoptions=gtk.EXPAND|gtk.FILL,
                     yoptions=gtk.EXPAND|gtk.FILL)

        combo.connect('changed', self._on_combo_position_changed)
        self.combo_position = combo
        self.label_position = label

    def _create_stock_widgets(self):
        "Create the widgets needed to setup a stock button"
        alignment = gtk.Alignment(1.0, 0.0, 1.0, 1.0)
        alignment.set_padding(0, 12, 48, 0)
        vbox = gtk.VBox()
        vbox.pack_start(self._create_stock_list())
        self.check_notext = gtk.CheckButton(_('No text'))
        self.check_notext.connect('toggled', self._on_check_notext_toggled)
        vbox.pack_start(self.check_notext, False, False)
        alignment.add(vbox)
        return alignment

    def _create_custom_widgets(self):
        "Create the widgets needed to setup a custom button"
        alignment = gtk.Alignment(1.0, 0.0, 1.0, 1.0)
        alignment.set_padding(0, 12, 48, 0)
        table = gtk.Table(rows=3, columns=2)
        table.set_col_spacings(6)
        table.set_row_spacings(2)
        
        self._create_label(table)

        self._create_image(table)
        
        self._create_position(table)
        
        alignment.add(table)

        return alignment
    
    def create(self):
        "Create the whole editor with all the widgets inside"
        
        mainvbox = gtk.VBox()

        # widgets to setup a custom button
        self.radio_custom = gtk.RadioButton(None, _('Use custom button:'))
        mainvbox.pack_start(self.radio_custom, False, False)

        self.custom_widgets = self._create_custom_widgets()
        mainvbox.pack_start(self.custom_widgets, True, True)

        # widgets to setup a stock button
        self.radio_stock = gtk.RadioButton(self.radio_custom,
                                           _('Use stock button:'))
        self.radio_stock.connect('toggled', self._on_radio_toggled)
        mainvbox.pack_start(self.radio_stock, False, False)
        self.stock_widgets = self._create_stock_widgets()
        mainvbox.pack_start(self.stock_widgets, True, True)

        # use an extra alignment to add some padding to the top
        al = gtk.Alignment(0.0, 0.0, 1.0, 1.0)
        al.set_padding(12, 0, 0, 0)
        al.add(mainvbox)
        al.show_all()
        return al

    def _on_radio_toggled(self, button):
        if self.loading: return
        
        # this is emitted when the stock radio is toggled
        value = self.radio_stock.get_active()
        self.stock_widgets.set_sensitive(value)
        self.custom_widgets.set_sensitive(not value)

        if value:
            # if there is nothing selected in the icon list select the first
            # item
            selection = self.stock_list.get_selection()
            model, sel_iter = selection.get_selected()
            if not sel_iter:
                # this is calling set_stock as a side effect
                selection.select_iter(model.get_iter_first())
            else:
                self.set_stock()
        else:
            self.set_custom()

    def _on_check_label_toggled(self, button):
        if self.loading: return
        
        value = self.check_label.get_active()
        self.entry_label.set_sensitive(value)
        self.button_label.set_sensitive(value)
        self._set_position_sensitive(value and self.check_image.get_active())

        self.set_custom()
        
    def _on_check_image_toggled(self, button):
        if self.loading: return
        
        value = self.check_image.get_active()
        self.entry_image.set_sensitive(value)
        self.button_image.set_sensitive(value)
        self._set_position_sensitive(value and self.check_label.get_active())

        self.set_custom()

    def _set_position_sensitive(self, value):
        self.label_position.set_sensitive(value)
        self.combo_position.set_sensitive(value)

    def _on_stock_list_selection_changed(self, selection):
        if not self.loading:
            self.set_stock()

    def _on_check_notext_toggled(self, button):
        if not self.loading:
            self.set_stock()
        
    def _on_entry_label_changed(self, entry):
        if not self.loading:
            self.set_custom()

    def _on_entry_image_changed(self, entry):
        if not self.loading:
            self.set_custom()

    def _on_button_label_clicked(self, button):
        pass
    
    def _on_button_image_clicked(self, button):
         dialog = gtk.FileChooserDialog(_('Open image ...'),
                                        self.application_window,
                                        gtk.FILE_CHOOSER_ACTION_OPEN,
                                        (gtk.STOCK_CANCEL,
                                         gtk.RESPONSE_CANCEL,
                                         gtk.STOCK_OPEN,
                                         gtk.RESPONSE_OK))
         file_filter = gtk.FileFilter()
         file_filter.set_name("Images")
         file_filter.add_mime_type("image/png")
         file_filter.add_mime_type("image/jpeg")
         file_filter.add_mime_type("image/gif")
         file_filter.add_mime_type("image/x-xpixmap")
         file_filter.add_pattern("*.png")
         file_filter.add_pattern("*.jpg")
         file_filter.add_pattern("*.gif")
         file_filter.add_pattern("*.xpm")

         dialog.add_filter(file_filter)
    
         response = dialog.run()
         filename = dialog.get_filename()
         if response == gtk.RESPONSE_OK and filename:
             self.entry_image.set_text(filename)

         dialog.destroy()

    def _on_combo_position_changed(self, combo):
        if not self.loading:
            self.set_custom()
            
    def set_stock(self):
        if not self.gwidget:
            return
        
        model, sel_iter = self.stock_list.get_selection().get_selected()
        if sel_iter:
            stock_id = model[sel_iter][COL_STOCK_ID]
        else:
            stock_id = None

        notext = self.check_notext.get_active()

        self.command_manager.set_button_contents(self.gwidget,
                                                 stock_id=stock_id,
                                                 notext=notext)

    def set_custom(self):
        if not self.gwidget:
            return
        
        label = None
        if self.check_label.get_active():
            label = self.entry_label.get_text() or None

        image = None
        if self.check_image.get_active():
            image = self.entry_image.get_text() or None
            
        position = None
        ac_iter = self.combo_position.get_active_iter()
        if ac_iter:
            model = self.combo_position.get_model()
            position = model[ac_iter][1]
            
        self.command_manager.set_button_contents(self.gwidget,
                                                 label=label,
                                                 image_path=image,
                                                 position=position)
    
        
    def _set_combo_active_position(self, pos):
        for row in self.combo_position.get_model():
            if row[1] == pos:
                self.combo_position.set_active_iter(row.iter)
                break

    def _set_stock_list_active_stock(self, stock_id):
        selection = self.stock_list.get_selection()
        for row in self.stock_list.get_model():
            if row[COL_STOCK_ID] == stock_id:
                selection.select_iter(row.iter)
                self.stock_list.scroll_to_cell(row.path, None,
                                               True, 0.5, 0.0)
                break
            
    def update(self, context, gtk_widget, proxy):
        self.command_manager = context.get_command_manager()
        self.gwidget = Widget.from_widget(gtk_widget)
        self.application_window = context.get_application_window()

        self.loading = True
        
        # default values for some widgets
        self.check_notext.set_active(False) 
        self.check_label.set_active(True)
        self.entry_label.set_text("")
        self.check_image.set_active(False)
        self.entry_image.set_text("")
        self.combo_position.set_active(0)
        self.combo_position.set_sensitive(False)

        (stock_id, notext, label,
         image_path, position) = util.get_button_state(gtk_widget)

        if stock_id:
            self.stock_widgets.set_sensitive(True)
            self.custom_widgets.set_sensitive(False)
            self.radio_stock.set_active(True)
            
            self._set_stock_list_active_stock(stock_id)
            self.check_notext.set_active(notext)

        else:
            self.stock_widgets.set_sensitive(False)
            self.custom_widgets.set_sensitive(True)
            self.radio_custom.set_active(True)
            
            self.check_label.set_active(True)
            self.entry_label.set_sensitive(True)
            self.button_label.set_sensitive(True)
            if label is not None:
                self.entry_label.set_text(label)

            if image_path is not None:
                self.check_image.set_active(True)
                self.entry_image.set_sensitive(True)
                self.entry_image.set_text(image_path)
            else:
                self.entry_image.set_sensitive(False)
                self.button_image.set_sensitive(False)

            if label and image_path:
                self.combo_position.set_sensitive(True)
                self._set_combo_active_position(position)
            else:
                self.combo_position.set_sensitive(False)

        self.loading = False

class ButtonContentsAdaptor(TransparentProperty, StringType):
    """This adaptor allow the user the choose between a stock button and
    a custom button.

    A stock button is a button with a stock-icon. The text is optional
    A custom button allow the user to add a label, an image and the position
    of the image with respect to the label.
    """
    editor = ButtonContentsEditor

    def get(self):
        return util.get_button_state(self._object)

    def save(self):
        return None

prop_registry.override_property('GtkButton::contents', ButtonContentsAdaptor)

class LabelAdaptor(CustomProperty, StringType):

    editable = False

    def is_translatable(self):
        # If a stock id is set, do not make it translatable,
        # because we don't want to translate stock-id labels!
        stock_id = util.get_button_state(self._object)[0]
        if stock_id:
            return False

        return True

    def save(self):
        (stock_id, notext, label,
         image_path, position) = util.get_button_state(self._object)
        if ((stock_id and notext) 
            or (not label and image_path)
            or (label and image_path)):
            return
        else:
            
            return self._object.get_property('label')
        
prop_registry.override_property('GtkButton::label', LabelAdaptor)

class ButtonAdaptor(ContainerAdaptor):
    def save(self, context, gwidget):
        """Create GazpachoWidgets for the possible children of
        this widget so they are saved with the button"""
        gtk_button = gwidget.gtk_widget
        
        (stock_id, notext, label,
         image_path, position) = util.get_button_state(gtk_button)

        child = gtk_button.get_child()
        project = context.get_project()

        if ((stock_id and notext)         # case 1: stock item without text
            or (not label and image_path) # case 2: only image
            or (label and image_path)):   # case 3: text and image

            self._create_names_for_internal_widgets(child, project)
            load_widget_from_gtk_widget(child, project)
            
            # case 4: stock item with text (nothing to do)
            # case 5: only text (nothing to do)

        # FIXME: Under some circumstances we end up with a GtkAlignment
        #        when we don't really need it.
        #        Find out when it's not needed and remove it

    def _create_names_for_internal_widgets(self, gtk_widget, project):
        class_name = gobject.type_name(gtk_widget)
        name = project.new_widget_name(class_name)
        gtk_widget.set_name(name)
        if isinstance(gtk_widget, gtk.Container):
            for child in gtk_widget.get_children():
                self._create_names_for_internal_widgets(child, project)

#TODO: we shouldn't need to specify these adaptors if adaptor inheritance would
# work
class HBoxAdaptor(ContainerAdaptor):
    pass

class VBoxAdaptor(ContainerAdaptor):
    pass

class HButtonBoxAdaptor(HBoxAdaptor):
    pass

class VButtonBoxAdaptor(VBoxAdaptor):
    pass

class TextViewAdaptor(ContainerAdaptor):
    pass

class StatusbarAdaptor(ContainerAdaptor):
    pass

class IconViewAdaptor(ContainerAdaptor):
    pass
    
# GtkWidget
class EventsProp(TransparentProperty):
    default = None
prop_registry.override_simple('GtkWidget::events',  EventsProp)

class VisibleProperty(CustomProperty, BooleanType):
    def get(self):
        value = self._object.get_data('gazpacho::visible')
        return value or False
    
    def set(self, value):
        self._object.set_data('gazpacho::visible', value)

prop_registry.override_property('GtkWidget::visible', VisibleProperty)
prop_registry.override_simple('GtkWidget::width-request',
                              minimum=-1, maximum=10000)
prop_registry.override_simple('GtkWidget::height-request',
                              minimum=-1, maximum=10000)

class TooltipProp(TransparentProperty):
    def get(self):
        data = gtk.tooltips_data_get(self._object)
        if data is not None:
            return data[2]
        
    def set(self, value):
        tooltips = self._project.tooltips
        if not tooltips:
            value = None
            
        tooltips.set_tip(self._object, value, None)
prop_registry.override_property('GtkWidget::tooltip', TooltipProp)

# GtkContainer
prop_registry.override_simple('GtkContainer::border-width',
                              minimum=0, maximum=10000)
prop_registry.override_simple('GtkContainer::resize-mode',
                              editable=False)

# GtkBox
class BoxSizeProp(CustomProperty, IntType):
    default = 3

    def get(self):
        return len(self._object.get_children())

    def set(self, new_size):
        old_size = len(self._object.get_children())
        if new_size == old_size:
            return
        elif new_size > old_size:
            # The box has grown. Add placeholders
            while new_size > old_size:
                self._object.add(Placeholder(self._project.get_app()))
                old_size += 1
        elif not isinstance(self._object, gtk.Statusbar):
            # The box has shrunk. Remove the widgets that are not on those slots
            child = self._object.get_children()[-1]
            while old_size > new_size and child:
                gwidget = Widget.from_widget(child)
                if gwidget: # It may be None, e.g a placeholder
                    gwidget.project.remove_widget(child)

                self._object.remove(child)
                child = self._object.get_children()[-1]
                old_size -= 1

    def save(self):
        return
    
prop_registry.override_property('GtkBox::size', BoxSizeProp)

# GtkMisc
align_ns = dict(default=0.5, minimum=0.0, maximum=1.0,
                step_increment=0.01, page_increment=0.1,
                climb_rate=0.2)
prop_registry.override_simple('GtkMisc::xalign', **align_ns)
prop_registry.override_simple('GtkMisc::yalign', **align_ns)

# GtkWindow
prop_registry.override_simple('GtkWindow::default-width',
                              minimum=-1, maximum=10000)
prop_registry.override_simple('GtkWindow::default-height',
                              minimum=-1, maximum=10000)
prop_registry.override_simple('GtkWindow::modal', TransparentProperty)
prop_registry.override_simple('GtkWindow::type-hint', TransparentProperty)
prop_registry.override_simple('GtkWindow::type', TransparentProperty,
                              editable=False)
prop_registry.override_simple('GtkWindow::allow-shrink', editable=False)
prop_registry.override_simple('GtkWindow::allow-grow', editable=False)
prop_registry.override_simple('GtkWindow::role', translatable=False)

# GtkLabel
prop_registry.override_simple('GtkLabel::mnemonic-widget', editable=True)

# GtkFrame
prop_registry.override_simple('GtkFrame::label', default="Frame")
prop_registry.override_simple('GtkFrame::label-xalign', **align_ns)
prop_registry.override_simple('GtkFrame::label-yalign', **align_ns)

# GtkExpander
prop_registry.override_simple('GtkExpander::expanded', default=True)

# GtkDialog
prop_registry.override_simple('GtkDialog::has-separator', default=False)

# GtkButton
prop_registry.override_simple('GtkButton::use-stock', editable=False)
#prop_registry.override_property('GtkButton::response-id', default="0", name="Response ID")

# GtkToggleButton
prop_registry.override_simple('GtkToggleButton::label', default="toggle button")

# GtkCheckButton
prop_registry.override_simple('GtkCheckButton::draw-indicator', default=True)
prop_registry.override_simple('GtkCheckButton::label', default="toggle button")

# GtkSpinButton
# <property id="adjustment.lower" name="Min" default="0">
# <tooltip>The minium value the spinbutton can have</tooltip>
# <property id="adjustment.upper" name="Max" default="100">
# <tooltip>The maximun value the spinbutton can have</tooltip>
# <property id="adjustment.step-increment" name="Step increment" default="1">
# <tooltip>Increment applied for each left mousebutton press</tooltip>
# <property id="adjustment.page-increment" name="Page increment" default="10">
# <tooltip>Increment applied for each middle mousebutton press</tooltip>

# GtkRadioButton
class RadioGroupProp(PropType):
    editable = True
    readable = True

    # We need to override get_default, since
    # the standard mechanism to get the initial value
    # is calling get_property, but ::group is not marked as readable
    def get_default(self, gobj):
        return
    
    def get(self):
        group_name = ''
        for group in self._object.get_group():
            if group.get_active():
                group_name = group.get_name()
        return group_name
    
    def set(self, group):
        if not group in self._object.get_group():
            self._object.set_property('group', group)

    def save(self):
        value = self.get()
        if value == self._object.get_name():
            return
        return value
        
prop_registry.override_property('GtkRadioButton::group', RadioGroupProp)

# GtkTable
prop_registry.override_simple('GtkTable::row-spacing',
                              minimum=0, maximum=10000)
prop_registry.override_simple('GtkTable::column-spacing',
                              minimum=0, maximum=10000)
#<child-properties>
#  <property id="left-attach" adaptor-class="LeftAttachAdaptor">
#      <parameter key="Min" value="0"/>
#      <parameter key="StepIncrement" value="1"/>
#      <parameter key="PageIncrement" value="1"/>
#    <function name="set" type="override"/>
#  <property id="right-attach" adaptor-class="RightAttachAdaptor">
#      <parameter key="Min" value="0"/>
#      <parameter key="StepIncrement" value="1"/>
#      <parameter key="PageIncrement" value="1"/>
#    <function name="set" type="override"/>
#  <property id="bottom-attach" adaptor-class="BottomAttachAdaptor">
#      <parameter key="Min" value="0"/>
#      <parameter key="StepIncrement" value="1"/>
#      <parameter key="PageIncrement" value="1"/>
#    <function name="set" type="override"/>
#  <property id="top-attach" adaptor-class="TopAttachAdaptor">
#      <parameter key="Min" value="0"/>
#      <parameter key="StepIncrement" value="1"/>
#      <parameter key="PageIncrement" value="1"/>
#    <function name="set" type="override"/>


# GtkComboBox
# These properties are buggie in GTK+ 2.4
prop_registry.override_simple('GtkComboBox::column-span-column', editable=False)
prop_registry.override_simple('GtkComboBox::row-span-column', editable=False)

# GtkComboBoxEntry
prop_registry.override_simple('GtkComboBoxEntry::text-column', default=0)

# GtkEntry
# optional="True" optional-default="False" ?
prop_registry.override_simple('GtkEntry::width-chars', default=0)

# GtkTextView
class TextProp(CustomProperty, StringType):
    def get(self):
        buffer = self._object.get_buffer()
        (start, end) = buffer.get_bounds()
        return buffer.get_text(start, end, False)

    def set(self, value):
        buffer = self._object.get_buffer()
        buffer.set_text(value or '')
prop_registry.override_property('GtkTextView::text', TextProp)

# GtkStatusbar
class HasResizeGripProp(CustomProperty, BooleanType):
    label = "Has Resize Grip"
    default = True
    def get(self):
        return self._object.get_has_resize_grip()

    def set(self, value):
        self._object.set_has_resize_grip(value)
prop_registry.override_simple('GtkStatusbar::has-resize-grip',
                              HasResizeGripProp)

# GtkNotebook

class NPagesProp(CustomProperty, IntType):
    minimum = 1
    maximum = 100
    step_increment = 1
    page_increment = 1
    climb_rate = 1
    label = "Number of pages"
    default = 3
    def get(self):
        return self._object.get_n_pages()

    def set(self, value):
        old_size = self._object.get_n_pages()
        new_size = value
        if new_size == old_size:
            return

        if new_size > old_size:
            # The notebook has grown. Add pages
            while new_size > old_size:
                label = None
                #print 'Create'
                #label = gtk.Label()
                #name = project.new_widget_name('label')
                #label.set_name(name)

                #print load_widget_from_gtk_widget(label, project)
                no = self._object.append_page(Placeholder(self._project.get_app()), label)
                #label.set_text(_('Page %d' % (no + 1)))
                #project.add_widget(label)
                old_size += 1
        else:
            # The notebook has shrunk. Remove pages

            # Thing to remember is that GtkNotebook starts the
            # page numbers from 0, not 1 (C-style). So we need to do
            # old_size-1, where we're referring to "nth" widget.
            while old_size > new_size:
                child_widget = self._object.get_nth_page(old_size - 1)
                child_gwidget = Widget.from_widget(child_widget)
                # If we got it, and it's not a placeholder, remove it from project
                if child_gwidget:
                    self._project.remove_widget(child_widget)

                self._object.remove_page(old_size - 1)
                old_size -= 1
prop_registry.override_property('GtkNotebook::n-pages', NPagesProp)

# GtkImage
class FileProp(CustomProperty, StringType):
    # we can remove this class when depending in gtk 2.8 since this property
    # is readable there
    editable = True
    readable = True
    translatable = False
    custom = True

    # We need to override get_default, since
    # the standard mechanism to get the initial value
    # is calling get_property, but ::file is not marked as readable
    def get_default(self, gobj):
        return ""

    def get(self):
        return self._object.get_data('image-file-name')

    def set(self, value):
        self._object.set_data('image-file-name', value)
        self._object.set_property('file', value)
    
prop_registry.override_property('GtkImage::file', FileProp)
prop_registry.override_simple('GtkImage::stock', default="gtk-missing-image",
                              translatable=False)

# GtkImage
# <property id="icon" name="Icon">
#   <function name="get" type="ignore"/>
#   <function name="set" type="ignore"/>
#   <function name="create" type="legacy" value="glade_gtk_image_icon_editor_create"/>
#   <function name="update" type="legacy" value="glade_gtk_image_icon_editor_update"/>
# <property id="icon-size">
#   <function name="create" type="legacy" value="glade_gtk_image_icon_size_editor_create"/>
#   <function name="update" type="legacy" value="glade_gtk_image_icon_size_editor_update"/>

# default child properties, not quite implemented
#prop_registry.override_simple('GtkLabel::expand', parent='GtkVBox', default=False)
#prop_registry.override_simple('GtkMenubar::expand', parent='GtkVBox', default=False)
#prop_registry.override_simple('GtkStatusbar::expand', parent='GtkVBox', default=False)
#prop_registry.override_simple('GtkToolbar::expand', parent='GtkVBox', default=False)
#prop_registry.override_simple('GtkVSeparator::expand', parent='GtkHBox', default=False)
