# Copyright (C) 2005 by Mattias Karlsson
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gobject
import gtk

from gazpacho.kiwiutils import gsignal, type_register

class GSizeGroup(gobject.GObject):

    gsignal('name-changed')
    gsignal('widgets-added', object)
    gsignal('widgets-removed', object)

    def __init__(self, name, group, widgets=[]):
        """
        Initialize the sizegroup.
        
        @param name: the name
        @type name: str
        @param group: sizegroup
        @type group: a gtk.SizeGroup
        """
        gobject.GObject.__init__(self)

        if not isinstance(group, gtk.SizeGroup):
            raise TypeError("group must be a gtk.SizeGroup, not %r" % group)
            
        self._name = name
        self._gtk_sizegroup = group
        self._widgets = widgets or [] # XXX: fixme

    #
    # Properties
    # 
    def _set_name(self, name):
        """
        Set the sizegroup name.

        @param name: the name
        @type name: str
        """
        self._name = name
        self.emit('name_changed')

    def _get_name(self):
        """
        Get the sizegroup name.

        @return: the sizegroup name
        @rtype: str
        """
        return self._name

    name = property(_get_name, _set_name)

    def _set_mode(self, mode):
        """
        Set the sizegroup mode.

        @param mode: the sizegroup mode
        @type mode: constants, see gtk.SizeGroup
        """
        self._gtk_sizegroup.set_mode(mode)
        self.emit('name-changed')

    def _get_mode(self):
        """
        Get the SizeGroup mode.

        @return: the sizegroup mode
        @rtype: constants, see gtk.SizeGroup
        """
        return self._gtk_sizegroup.get_mode()

    mode = property(_get_mode, _set_mode)

    #
    # Public methods
    # 
    def is_empty(self):
        """
        Check if the sizegroup is empty

        @return: true if the sizegroup doesn't have any widgets
        @rtype: bool
        """
        return not self._widgets

    def has_widget(self, widget):
        """
        Check if the size group has the widget.

        @param widget: the widget to test
        @type widget: gazpacho.widget.Widget
        
        @return: true if the widget is in the group
        @rtype: bool
        """
        return widget in self._widgets

    def get_widgets(self):
        """
        Get the widgets for this group.

        @return: the widgets
        @rtype: list (of gazpacho.widget.Widget)
        """
        return self._widgets[:]
    
    def add_widgets(self, widgets):
        """
        Add widgets to the SizeGroup.

        @param widgets: the widgets to add
        @type widgets: list (of gazpacho.widget.Widget)
        """
        if not widgets:
            return
        
        self._widgets += widgets
        for widget in widgets:
            prop = widget.get_prop('sizegroup')
            prop.value = self._name
            self._add_gtk_widget(widget.gtk_widget)

        self.emit('widgets-added', widgets)

    def remove_widgets(self, widgets):
        """
        Remove widgets from the sizegroup.

        @param widgets: the widgets to remove
        @type widgets: list (of gazpacho.widget.Widget)
        """
        if not widgets:
            return
        
        for widget in widgets:
            self._widgets.remove(widget)
            self._remove_gtk_widget(widget.gtk_widget)

        self.emit('widgets-removed', widgets)

    #
    # Private methods
    # 
    def _add_gtk_widget(self, gtk_widget):
        """
        Add a gtk widget to the gtk SizeGroup.

        @param gtk_widget: the gtk widget
        @type gtk_widget: gtk.Widget
        """
        self._gtk_sizegroup.add_widget(gtk_widget)

    def _remove_gtk_widget(self, gtk_widget):
        """
        Remove a gtk widget from the gtk SizeGroup.

        @param gtk_widget: the gtk widget
        @type gtk_widget: gtk.Widget
        """
        self._gtk_sizegroup.remove_widget(gtk_widget)
        
type_register(GSizeGroup)

def safe_to_add_widget(sizegroup, widget):
    """
    Test if it is safe to add the widget to the sizegroup. Widget who
    has an ancestor or child in the sizegroup cannot be added. Note
    that it is concidered ok to add widgets that are already in the
    sizegroup since they will just be ignored.
    """

    # Create widget parent cache
    cache = {}
    for sizegroup_widget in sizegroup.get_widgets():
        parent = sizegroup_widget.get_parent()
        while parent:
            if parent in cache:
                break
            cache[parent] = parent
            parent = parent.get_parent()

    # We're not allowed to add a widget that is an ancestor to a
    # widget in the sizegroup
    if widget in cache:
        return False

    # We're not allowed to add a widet who has an ancestor in the
    # sizegroup
    parent = widget.get_parent()
    while parent:
        if sizegroup.has_widget(parent):
            return False
        parent = parent.get_parent()
        
    return True
