# Copyright (C) 2004,2005 by SICEm S.L. and Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

import gettext

import gobject
import gtk

from gazpacho import util
from gazpacho.app.bars import bar_manager
from gazpacho.annotator import Annotator
from gazpacho.commandmanager import command_manager
from gazpacho.dndhandlers import WidgetDnDHandler
from gazpacho.kiwiutils import type_register
from gazpacho.loader import tags
from gazpacho.popup import Popup
from gazpacho.placeholder import Placeholder
from gazpacho.properties import prop_registry
from gazpacho.widgetregistry import widget_registry

_ = gettext.gettext

def copy_widget(source_gadget, project, keep_name=False):
    """Create a copy of the widget. You must specify a project since
    a name conflict resolution will be performed for the copy."""

    from gazpacho.project import GazpachoObjectBuilder

    # Create new widget
    wt = GazpachoObjectBuilder(buffer=source_gadget.toxml(),
                               placeholder=Placeholder)
    widget = wt.get_widget(source_gadget.name)
    gadget_copy = load_widget_from_widget(widget, project)

    if not keep_name:
        project.set_new_widget_name(source_gadget.widget)

    return gadget_copy

def load_widget_from_xml(xml_data, project):
    """Load a widget from an XML representation of that widget. If
    there are multiple root widgets in the XML none of them will be
    created."""
    from gazpacho.project import GazpachoObjectBuilder

    wt = GazpachoObjectBuilder(buffer=xml_data, placeholder=Placeholder)
    widgets = wt.get_widgets()

    if not widgets or len(widgets) > 1:
        # We fail if none or too many widgets were found
        return

    return load_widget_from_widget(widgets[0], project)

def load_widget_from_widget(widget, project):
    """  This function creates a gadget from a widget.
    It recursively creates any children of the original widget.
    This method is mainly used when loading a .glade file with
    gazpacho.loader

    Also makes sure the name of the widget is unique in the project
    parameter and if not, change it for a valid one.
    """

    # widget may not have a name if it is an intermediate internal children
    # For example, the label of a button does not have a name
    if widget.name is None:
        # this is not an important widget but maybe its children are
        if isinstance(widget, gtk.Container):
            for child in widget.get_children():
                load_widget_from_widget(child, project)
        return

    type_name = gobject.type_name(widget)
    adaptor = widget_registry.get_by_name(type_name)
    if adaptor is None:
        print ('Warning: could not get the class from widget: %s' %
               widget)
        return

    context = project.context
    return adaptor.load(context, widget)

class Gadget(gobject.GObject):
    """
    This is essentially a wrapper around regular GtkWidgets with more data and
    functionality needed for Gazpacho.

    It has access to the GtkWidget it is associated with, the Project it
    belongs to and the instance of its WidgetAdaptor.

    Given a GtkWidget you can ask which (Gazpacho) Widget is associated with it
    using the get_from_widget function or, if no one has been created yet you
    can create one with the create_from_widget method.

    A Widget have a signal dictionary where the keys are signal names and the
    values are lists of signal handler names. There are methods to
    add/remove/edit these.

    It store all the Properies (wrappers for GtkWidgets properties) and you can
    get any of them with get_prop. It knows how to handle normal
    properties and packing properties.
    """

    def __init__(self, adaptor, project):
        """This init method is not complete because there are several scenarios
        where widgets are created:

            - Normal creation in Gazpacho
            - When loading from a .glade file
            - When copying from another widget
            - Internal widgets

        So you should call the appropiate method after this basic initialization.
        """
        gobject.GObject.__init__(self)

        # If the widget is an internal child of another widget this is the
        # name of the internal child, otherwise is None
        # Internal children can not be deleted
        self.internal_name = None

        self.properties = {}

        # The WidgetAdaptor this widget is an instance of
        self.adaptor = adaptor

        # The Project this widget belongs to
        self.project = project

        # The GtkWidget associated with this Widget
        self.widget = None

        # Constructor, used by UIManger
        self.constructor = None

        # DND the widget that should be draged (not necessarily this one)
        self.dnd_widget = None

        # flag that tells the property code if this widget should keep the
        # values of its properties when creating the GazpachoProperties or not
        self.maintain_gtk_properties = False

        # A dictionary of signals (signal, handlers) indexed by their names
        self._signals = {}
        self._dnd_drop_region = None

    def __repr__(self):
        if self.widget:
            name = self.widget.name
        else:
            name = '(empty)'
        return '<Gadget %s>' % name

    def create_widget(self, interactive=True):
        """Second part of the init process when creating a widget in
        the usual way.
        If the 'interactive' argument is false no popups will show up to
        ask for options
        """

        # XXX: Clean up this horrible mess
        widget = self.adaptor.create(self.project.context, interactive)
        self.adaptor.pre_create(self.project.context, widget, interactive)
        self.project.set_new_widget_name(widget)
        self.setup_widget(widget)
        self.adaptor.post_create(self.project.context, widget, interactive)
        self.adaptor.fill_empty(self.project.context, widget)

    def load(cls, widget, project):
        """Helper function of load_widget_from_widget. Create the Gazpacho
        widget and create a right name for it
        """
        assert project != None

        adaptor = widget_registry.get_by_type(widget)
        assert adaptor != None, widget

        gadget = cls(adaptor, project)
        gadget.setup_widget(widget)

        gadget.internal_name = widget.get_data(
            'gazpacho::internal-child-name')

        return gadget
    load = classmethod(load)

    def replace(cls, old_widget, new_widget, parent):
        gnew_gadget = cls.from_widget(new_widget)
        gold_gadget = old_widget and cls.from_widget(old_widget)

        gtk_new_widget = gnew_gadget and gnew_gadget.widget or new_widget
        gtk_old_widget = gold_gadget and gold_gadget.widget or old_widget

        if parent is None:
            parent = util.get_parent(old_widget)

        parent.adaptor.replace_child(parent.project.context,
                                     gtk_old_widget,
                                     gtk_new_widget,
                                     parent.widget)
        if gnew_gadget:
            gnew_gadget._connect_signal_handlers(gnew_gadget.widget)

    replace = classmethod(replace)

    def from_widget(cls, widget):
        return widget.get_data('gazpacho::gadget')
    from_widget = classmethod(from_widget)

    def setup_toplevel(self):
        """Add the action groups of Gazpacho to this toplevel and
        also set this toplevel transient for the main window.
        """
        widget = self.widget
        widget.add_accel_group(bar_manager.get_accel_group())

        # make window management easier by making created windows
        # transient for the editor window
        widget.set_transient_for(self.project._app.window)

    def setup_widget(self, widget):
        if self.widget is not None:
            self.widget.set_data('gazpacho::gadget', None)
            self.widget = None

        self.widget = widget
        self.widget.set_data('gazpacho::gadget', self)
        self.widget.add_events(gtk.gdk.BUTTON_PRESS_MASK |
                                   gtk.gdk.BUTTON_RELEASE_MASK |
                                   gtk.gdk.KEY_PRESS_MASK |
                                   gtk.gdk.POINTER_MOTION_MASK)
        if self.widget.flags() & gtk.TOPLEVEL:
            self.widget.connect('delete_event', self._hide_on_delete)

        self.widget.connect('popup_menu', self._popup_menu)
        self.widget.connect('key_press_event', self._key_press)

        self._connect_signal_handlers(widget)

        # Init Drag and Drop
        self.dndhandler = WidgetDnDHandler()
        self.dndhandler.connect_drag_handlers(self.widget)
        if not isinstance(self.widget, gtk.Container):
            self.dndhandler.connect_drop_handlers(self.widget)

    def set_drop_region(self, region):
        """Set the drop region and make sure it is painted.

        @param region: a tuple of x, y, width and height
        @type region: tuple
        """
        self._dnd_drop_region = region
        self.widget.queue_draw()

    def clear_drop_region(self):
        """Clear the drop region and make sure it is not painted anymore."""
        if self._dnd_drop_region:
            self._dnd_drop_region = None
            self.widget.queue_draw()

    def  _draw_drop_region(self, widget, expose_window, annotator=None):
        if not annotator:
            annotator = Annotator(widget, expose_window)

        color = (0, 0, 1)

        (x, y, width, height) = self._dnd_drop_region

        annotator.draw_border(x, y, width, height, color)

    def setup_internal_widget(self, widget, internal_name, parent_name):
        self.internal_name = internal_name
        widget.set_name(parent_name + '-' + internal_name)

        self.setup_widget(widget)

    # Properties

    def _get_name(self):
        return self.widget.name

    def _set_name(self, value):
        # set_name calls notify::name
        self.widget.set_name(value)

    name = property(_get_name, _set_name)

    # utility getter methods
    def get_parent(self):
        if self.widget.flags() & gtk.TOPLEVEL:
            return None

        return util.get_parent(self.widget)

    def add_signal_handler(self, signal_handler):
        self._signals.setdefault(signal_handler.name, []).append(
            signal_handler)

    def remove_signal_handler(self, signal_handler):
        name = signal_handler.name
        if name in self._signals:
            self._signals[name].remove(signal_handler)

    def change_signal_handler(self, old_signal_handler, new_signal_handler):
        old_name = old_signal_handler.name
        if old_name != new_signal_handler.name:
            return

        if not old_name in self._signals:
            return

        signals = self._signals[old_name]
        signals[signals.index(old_signal_handler)] = new_signal_handler.copy()

    def _expose_event_after(self, widget, event):
        util.draw_annotations(widget, event.window)

        if self._dnd_drop_region:
            self._draw_drop_region(widget, event.window)

    def _find_deepest_child_at_position(self, toplevel, container, x, y):
        found = None
        if isinstance(container, gtk.Container):
            for widget in container.get_children():
                coords = toplevel.translate_coordinates(widget, x, y)
                if len(coords) != 2:
                    continue
                child_x, child_y = coords

                # sometimes the found widget is not mapped or visible
                # think about a widget in a notebook page which is not selected
                if (0 <= child_x < widget.allocation.width and
                    0 <= child_y < widget.allocation.height and
                    widget.flags() & (gtk.MAPPED | gtk.VISIBLE)):
                    found = self._find_deepest_child_at_position(toplevel,
                                                                 widget, x, y)
                    if found:
                        break

        return found or Gadget.from_widget(container)

    def _retrieve_from_event(self, base, event):
        """ Returns the real GtkWidget in x, y of base.
        This is needed because for GtkWidgets than does not have a
        GdkWindow the click event goes right to their parent.
        """

        x = int(event.x)
        y = int(event.y)
        window = base.get_toplevel()
        if window.flags() & gtk.TOPLEVEL:
            top_x, top_y = base.translate_coordinates(window, x, y)
            return self._find_deepest_child_at_position(window, window,
                                                        top_x, top_y)

    def _event(self, widget, event):
        """We only delegate this call to the appropiate event handler"""
        if event.type == gtk.gdk.BUTTON_PRESS:
            return self._button_press_event(widget, event)
        elif event.type == gtk.gdk.BUTTON_RELEASE:
            return self._button_release_event(widget, event)
        elif event.type == gtk.gdk.MOTION_NOTIFY:
            return self._motion_notify_event(widget, event)

        return False

    def _event_after(self, widget, event):
        """We only delegate this call to the appropiate event handler"""
        if event.type == gtk.gdk.EXPOSE:
            self._expose_event_after(widget, event)

    def _motion_notify_event(self, widget, event):
        gadget = self._retrieve_from_event(widget, event)
        if not gadget:
            return
        return gadget.adaptor.motion_notify(gadget.project.context,
                                             gadget.widget, event)

    def _button_release_event(self, widget, event):
        gadget = self._retrieve_from_event(widget, event)
        if not gadget:
            return
        return gadget.adaptor.button_release(gadget.project.context,
                                              gadget.widget, event)

    def _button_press_event(self, widget, event):
        gadget = self._retrieve_from_event(widget, event)
        if not gadget:
            return

        # DND store the widget, we might need it later
        self.dnd_widget = gadget

        widget = gadget.widget
        # Make sure to grab focus, since we may stop default handlers
        if widget.flags() & gtk.CAN_FOCUS:
            widget.grab_focus()

        if event.button == 1:
            if event.type is not gtk.gdk.BUTTON_PRESS:
                #only single clicks allowed, thanks
                return False

            # Shift clicking circles through the widget tree by
            # choosing the parent of the currently selected widget.
            if event.state & gtk.gdk.SHIFT_MASK:
                util.circle_select(widget)
                return True

            # if it's already selected don't stop default handlers,
            # e.g toggle button
            selected = util.has_nodes(widget)
            self.project.selection_set(widget, True)
            return not selected
        elif event.button == 3:
            # first select the widget
            self.project.selection_set(widget, True)

            # then popup the menu
            popup = Popup(command_manager, gadget)
            popup.pop(event)
            return True

        return False

    def _popup_menu(self, widget):
        return True # XXX TODO

    def _key_press(self, widget, event):
        gadget = Gadget.from_widget(widget)

        if event.keyval in (gtk.keysyms.Delete, gtk.keysyms.KP_Delete):
            # We will delete all the selected items
            gadget.project.delete_selection()
            return True

        return False

    def _hide_on_delete(self, widget, event):
        return widget.hide_on_delete()

    def _connect_signal_handlers(self, widget):
        # don't connect handlers for placeholders
        if isinstance(widget, Placeholder):
            return

        widget.set_redraw_on_allocate(True)

        # check if we've already connected an event handler
        if not widget.get_data(tags.EVENT_HANDLER_CONNECTED):
            # we are connecting to the event signal instead of the more
            # apropiated expose-event and button-press-event because some
            # widgets (ComboBox) does not allows us to connect to those
            # signals. See http://bugzilla.gnome.org/show_bug.cgi?id=171125
            # Hopefully we will be able to fix this hack if GTK+ is changed
            # in this case
            widget.connect('event', self._event)
            widget.connect('event-after', self._event_after)

#            widget.connect("expose-event", self._expose_event)
#            widget.connect("button-press-event", self._button_press_event)
            widget.set_data(tags.EVENT_HANDLER_CONNECTED, 1)

        ##
        ## # we also need to get expose events for any children
        ## if isinstance(widget, gtk.Container):
        ##     widget.forall(self._connect_signal_handlers)

    def is_toplevel(self):
        return self.adaptor.is_toplevel()

    def list_signal_handlers(self, signal_name):
        return self._signals.get(signal_name, [])

    def get_all_signal_handlers(self):
        return self._signals.values()

    def _create_prop(self, prop_name):
        # Get the property type
        type_name = gobject.type_name(self.widget)
        parent = self.widget.get_parent()
        prop_type = prop_registry.get_prop_type(type_name, prop_name,
                                                parent=parent)
        # Instantiate the property from the property type
        return prop_type(self)

    def get_prop(self, prop_name):
        properties = self.properties
        if prop_name in properties:
            return properties[prop_name]

        prop = self._create_prop(prop_name)
        properties[prop_name] = prop
        return prop

    def toxml(self):
        from gazpacho.filewriter import XMLWriter
        xw = XMLWriter(project=self.project)
        return xw.serialize(self)

type_register(Gadget)

