/***************************************************************
* C file: cpuid.c... for CPUSPEED.EXE
*
*       This program has been developed by Intel Corporation.  
*		You have Intel's permission to incorporate this code 
*       into your product, royalty free.  Intel has various 
*	    intellectual property rights which it may assert under
*       certain circumstances, such as if another manufacturer's
*       processor mis-identifies itself as being "GenuineIntel"
*		when the CPUID instruction is executed.
*
*       Intel specifically disclaims all warranties, express or
*       implied, and all liability, including consequential and
*		other indirect damages, for the use of this code, 
*		including liability for infringement of any proprietary
*		rights, and including the warranties of merchantability
*		and fitness for a particular purpose.  Intel does not 
*		assume any responsibility for any errors which may 
*		appear in this code nor any responsibility to update it.
*
*  * Other brands and names are the property of their respective
*    owners.
*
*  Copyright (c) 1995, Intel Corporation.  All rights reserved.
***************************************************************/

#include "cpuid.h"
#include "cpuspeed.h"

/*#define CPU_ID _asm _emit 0x0f _asm _emit 0xa2  // CPUID opcode
#define OPND32 _asm _emit 0x66	// Macro for allowing 32-bit 
                        //   in-line assembly opcodes*/


// Global Variable /////////////////////////////////////////////
int clone_flag;				// Flag to show whether processor
							//   is an Intel clone



// Public Functions ////////////////////////////////////////////

/***************************************************************
* cpuidsupport()
*
* Inputs: none
*
* Returns:
*  1 = CPUID opcode is supported
*  0 = CPUID opcode is not supported
***************************************************************/

int cpuidsupport() {
   int cpuid_support = 1;
	_asm {
      OPND32
      pushf					// Get original EFLAGS
      OPND32
      pop ax
      OPND32
      mov cx, ax
      // xor     ax, 200000h // flip ID bit in EFLAGS
		// the following 7 lines replace the xor above
		mov dx, ax
		OPND32
		shr ax, 16
		xor ax, 0020h
		OPND32
		shl ax, 16
		mov ax, dx
		/***********************************************/
      OPND32
      push ax                      // Save new EFLAGS value on stack
		OPND32
      popf                         // Replace current EFLAGS value
		OPND32
      pushf                        // Get new EFLAGS
		OPND32
      pop ax                       // Store new EFLAGS in EAX
		OPND32
      xor ax, cx        	        // Can't toggle AC bit,
      jnz support
		mov cpuid_support,0
   }
support:
   return cpuid_support;
} // cpuidsupport



/***************************************************************
* cpuid()
*
* Inputs: none
*
* Returns:
* -1 = Error occurred, or a processor beyond the 80486 but not 
*         genuine Intel.
*  0 = 8086/88
*  2 = 80286
*  3 = 80386
*  4 = 80486
*  5 = Pentium(R) Processor
*  6 = PentiumPro(R) Processor
*  7 or higher = Processor beyond the PentiumPro6(R) Processor
***************************************************************/

int cpuid (void) {
   int cpuid = 0xffff;
   int cpuid_support = cpuidsupport ();
   if (cpuid_support)
      cpuid = check_IDProc ();
   else {
      clone_flag = check_clone ();
      cpuid = check_8086 ();            // Will return FFFFh or 0
      if (cpuid == 0)
         goto end;
      cpuid = check_80286 ();           // Will return FFFFh or 2
      if (cpuid == 2)
         goto end;
      cpuid = check_80386 ();           // Will return FFFFh or 3
      if (cpuid == 3)
         goto end;                      // temporarily commented out.
      cpuid = 4;  /* If the processor does not support CPUID,
                     is not an 8086, 80286, or 80386, assign
                     processor to be an 80486 */
   }
end:
	if (clone_flag)
		cpuid = cpuid | CLONE_MASK;	   // Signify that a clone has been
		                                 //   detected by setting MSB high
   return cpuid;
} // cpuid()
// Internal Private Functions //////////////////////////////////

/***************************************************************
* check_clone()
*
* Inputs: none
*
* Returns:
*   1      if processor is clone (limited detection ability)
*   0      otherwise
***************************************************************/

int check_clone (void) {
	short cpu_type = 0;
	asm {
  	   mov ax, 0x5555            // Check to make sure it is a 386
	   xor dx, dx                // To detect the NexGen processor,  first do
	   mov cx, 0x2               // check, whether it is an i80386 compatible
	   div cx                    // processor, or not. If it is a 32 bit CPU,
	   clc                       // then execute the division,  which changes
	   jnz no_clone              // zero flag on non-NexGen processors.  Only
	   jmp clone                 // the NexGen will not change the zero flag.
   }
no_clone:
   asm stc;
clone:
   asm {
      pushf                     // C=0 is a NexGen, C=1 is no NexGen
		pop ax                    // get the flags (carry is used as a 'flag')
		and al,1                  // mask the carry flag and negate it
	   xor al,1                  // AL=0 is no NexGen, AL=1 is a NexGen
		mov cpu_type, ax
	}
   cpu_type = cpu_type & 0x0001;
	return cpu_type;
} // check_clone()



/***************************************************************
* check_8086()
*
* Inputs: none
*
* Returns:
*   0      if processor 8086
*   0xffff otherwise
***************************************************************/

int check_8086 (void) {
   short cpu_type = 0xffff;
   _asm {
      pushf                   // Push original FLAGS
      pop     ax              // Get original FLAGS
      mov     cx, ax          // Save original FLAGS
      and     ax, 0fffh       // Clear bits 12-15 in FLAGS
      push    ax              // Save new FLAGS value on stack
      popf                    // Replace current FLAGS value
      pushf                   // Get new FLAGS
      pop     ax              // Store new FLAGS in AX
      and     ax, 0f000h      // If bits 12-15 are set, then
      cmp     ax, 0f000h      //   processor is an 8086/8088
      mov cpu_type, 0         // Turn on 8086/8088 flag
      je end_8086             // Jump if processor is 8086/8088
      mov cpu_type, 0ffffh
   }
end_8086:
   asm {
		push 	cx
		popf
		mov		ax, cpu_type
   }
	return cpu_type;
} // check_8086()



/***************************************************************
* check_80286()
*
* Inputs: none
*
* Returns:
*   2      if processor 80286
*   0xffff otherwise
***************************************************************/

int check_80286()
{

		int cpu_type=0xffff;

_asm {
		pushf
		pop		cx
		mov		bx, cx
        or      cx, 0f000h      // Try to set bits 12-15
        push    cx              // Save new FLAGS value on stack
        popf                    // Replace current FLAGS value
        pushf                   // Get new FLAGS
        pop     ax              // Store new FLAGS in AX
        and     ax, 0f000h      // If bits 12-15 are clear
        
        mov     cpu_type, 2     // Processor=80286, turn on 
        						//   80286 flag
        
        jz      end_80286       // If no bits set, processor is 
        						//   80286
		
		mov		cpu_type, 0ffffh
      }
end_80286:
asm {
		push	bx
		popf
		mov		ax, cpu_type

      }
	return cpu_type;
} // check_80286()



/***************************************************************
* check_80386()
*
* Inputs: none
*
* Returns:
*   3      if processor 80386
*   0xffff otherwise
***************************************************************/

int check_80386()
{

		int cpu_type=0xffff;

_asm {
		mov 	bx, sp
		and		sp, not 3
		OPND32
        pushf					// Push original EFLAGS
		OPND32
        pop     ax				// Get original EFLAGS
		OPND32
        mov     cx, ax			// Save original EFLAGS
        
        //xor     ax, 40000h     ; flip AC bit in EFLAGS
		// the following 7 lines replace the xor above
		
		mov		dx, ax
		OPND32
		shr		ax, 16
		xor		ax, 0004h
		OPND32
		shl		ax, 16
		mov		ax, dx
		/***********************************************/

		OPND32
        push    ax              // Save new EFLAGS value on 
        						//   stack
		OPND32
        popf                    // Replace current EFLAGS value
		OPND32
        pushf					// Get new EFLAGS
		OPND32
        pop     ax				// Store new EFLAGS in EAX
		OPND32
        xor     ax, cx        	// Can't toggle AC bit, 
        						//   processor=80386
        
        mov     cpu_type, 3		// Turn on 80386 processor flag
        jz      end_80386    	// Jump if 80386 processor
		mov		cpu_type, 0ffffh
   }
end_80386:
   asm {
		OPND32
		push	cx
		OPND32    
		popf
		mov		sp, bx
		mov		ax, cpu_type
		OPND32
		and		ax, 0000ffffh
      }
	return cpu_type;
} // check_80386()



/***************************************************************
* check_IDProc()
*
* Inputs: none
*
* Returns:
*  5 if Pentium(R) Processor
*  6 or higher if Processor beyond the Pentium(R) Processor
***************************************************************/

int check_IDProc()		// Processor can execute CPUID op; may 
						//   be 80486 or greater
{

		int i=0;
		int cpu_type=0xffff;
		char cpuid_flag=0;
		char stepping=0;
		char model=0;
		char  vendor_id[12]= "------------";
		char  intel_id[12] = "GenuineIntel";

_asm {

        mov     cpuid_flag, 1	// Flag indicating use of CPUID
        						//   instruction
        OPND32
        xor     ax, ax          // Set up for CPUID instruction
        CPU_ID                  // Get and save vendor ID

		OPND32
        mov     word ptr vendor_id, bx
        OPND32
        mov     word ptr vendor_id[+4], dx
        OPND32
        mov     word ptr vendor_id[+8], cx
}

for (i=0;i<12;i++)
{
	if (!(vendor_id[i]==intel_id[i]))
		clone_flag = 1;
}

_asm {
		OPND32
        cmp     ax, 1			// Make sure 1 is valid input 
        						//   for CPUID
        
        jl      end_IDProc		// If not, jump to end
		OPND32
        xor     ax, ax
        OPND32
        inc		ax
        CPU_ID					// Get family/model/stepping/
        						//   features

		mov 	stepping, al
		and		stepping, 0fh

		and 	al, 0f0h
		shr		al, 4
		mov 	model, al

		and		ax, 0f00h
        shr     ax, 8			// Isolate family
		and		ax, 0fh
        mov     cpu_type, ax	// Set cpu_type with family
   }
end_IDProc:
   asm mov ax, cpu_type
	return cpu_type;
} // check_IDPRoc()



